<?php
// /admin/api/newsletter/add-prospect.php
// Endpoint pour ajouter un nouveau prospect depuis N8N

// Nettoyer le buffer de sortie pour éviter les caractères parasites
ob_start();

require_once '../../includes/config.php';

// Headers avec charset explicite
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: https://bio360.idkom.fr');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Vérification méthode POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    ob_clean();
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Authentification API simple
$auth_header = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
$expected_token = 'n8n_api_token_2025';
if ($auth_header !== 'Bearer ' . $expected_token) {
    http_response_code(401);
    ob_clean();
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

// Récupération des données JSON
$json = file_get_contents('php://input');
$data = json_decode($json, true);

if (!$data) {
    http_response_code(400);
    ob_clean();
    echo json_encode(['error' => 'Invalid JSON']);
    exit;
}

// Validation des champs requis
$required_fields = ['company', 'city'];
foreach ($required_fields as $field) {
    if (empty($data[$field])) {
        http_response_code(400);
        ob_clean();
        echo json_encode(['error' => "Missing required field: $field"]);
        exit;
    }
}

// Nettoyage et validation des données
$email = !empty($data['email']) ? filter_var(trim($data['email']), FILTER_VALIDATE_EMAIL) : null;
if (!empty($data['email']) && !$email) {
    http_response_code(400);
    ob_clean();
    echo json_encode(['error' => 'Invalid email format']);
    exit;
}

$company = trim($data['company']);
$city = trim($data['city']);
$source = $data['source'] ?? 'n8n_api';
$industry = $data['industry'] ?? 'Événementiel';

try {
    // Vérifier si l'email existe déjà
    $check_stmt = $db->prepare("SELECT id FROM newsletter_contacts WHERE email = ?");
    $check_stmt->execute([$email]);
    
    if ($check_stmt->fetch()) {
        // Email existe déjà, mise à jour des infos
        $update_stmt = $db->prepare("
            UPDATE newsletter_contacts 
            SET company = ?, city = ?, industry = ?, source = ?, updated_at = NOW()
            WHERE email = ?
        ");
        $update_stmt->execute([$company, $city, $industry, $source, $email]);
        
        $response = [
            'success' => true,
            'action' => 'updated',
            'email' => $email,
            'message' => 'Contact mis à jour'
        ];
    } else {
        // Nouvel email, insertion
        $insert_stmt = $db->prepare("
            INSERT INTO newsletter_contacts 
            (email, company, city, industry, source, consent_status, created_at, updated_at)
            VALUES (?, ?, ?, ?, ?, 'b2b_cold', NOW(), NOW())
        ");
        $insert_stmt->execute([$email, $company, $city, $industry, $source]);
        
        $response = [
            'success' => true,
            'action' => 'created',
            'email' => $email,
            'id' => (int)$db->lastInsertId(),
            'message' => 'Nouveau contact ajouté'
        ];
    }
    
    // Nettoyer le buffer et envoyer la réponse JSON
    ob_clean();
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    
} catch (PDOException $e) {
    http_response_code(500);
    ob_clean();
    echo json_encode([
        'error' => 'Database error',
        'message' => 'Erreur lors de l\'insertion'
    ], JSON_UNESCAPED_UNICODE);
    error_log("Newsletter API Error: " . $e->getMessage());
}

// Nettoyer le buffer de sortie final
ob_end_flush();
?>