<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once dirname(__FILE__) . '/../../../includes/config.php';
require_once dirname(__FILE__) . '/../../../includes/auth.php';

$page_title = 'Nouvelle Campagne Newsletter';
requireAdminAuth();

// Traitement du formulaire
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $result = createCampaign($db, $_POST);
    if ($result['success']) {
        header('Location: index.php?created=' . $result['campaign_id']);
        exit;
    } else {
        $error = $result['message'];
    }
}

// Récupération des données nécessaires
$templates = getTemplates($db);
$segments = getSegments($db);
$contactsCount = getContactsCount($db);

require_once dirname(__FILE__) . '/../../../includes/header.php';
?>

<style>
.create-header {
    background: linear-gradient(135deg, #17a2b8 0%, #6f42c1 100%);
    color: white;
    padding: 2rem 0;
    margin-bottom: 2rem;
}

.step-indicator {
    display: flex;
    justify-content: center;
    margin-bottom: 2rem;
}

.step {
    display: flex;
    align-items: center;
    margin: 0 1rem;
}

.step-number {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #6c757d;
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    margin-right: 0.5rem;
}

.step.active .step-number {
    background: #0d6efd;
}

.step.completed .step-number {
    background: #198754;
}

.form-section {
    display: none;
}

.form-section.active {
    display: block;
}

.template-preview {
    border: 2px solid #dee2e6;
    border-radius: 8px;
    padding: 1rem;
    margin-bottom: 1rem;
    cursor: pointer;
    transition: all 0.3s ease;
}

.template-preview:hover {
    border-color: #0d6efd;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.template-preview.selected {
    border-color: #0d6efd;
    background: #f8f9ff;
}

.audience-stats {
    background: #f8f9fa;
    border-radius: 8px;
    padding: 1rem;
    margin-top: 1rem;
}

.preview-container {
    border: 1px solid #dee2e6;
    border-radius: 8px;
    background: white;
    min-height: 400px;
}
</style>

<div class="create-header">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h1><i class="fas fa-plus-circle me-3"></i>Nouvelle Campagne</h1>
                <p class="mb-0">Créez votre campagne newsletter en quelques étapes</p>
            </div>
            <div class="col-md-4 text-end">
                <a href="index.php" class="btn btn-light">
                    <i class="fas fa-arrow-left me-2"></i>Retour aux campagnes
                </a>
            </div>
        </div>
    </div>
</div>

<div class="container">
    <?php if (isset($error)): ?>
        <div class="alert alert-danger">
            <i class="fas fa-exclamation-triangle me-2"></i><?php echo htmlspecialchars($error); ?>
        </div>
    <?php endif; ?>

    <!-- Indicateur d'étapes -->
    <div class="step-indicator">
        <div class="step active" data-step="1">
            <div class="step-number">1</div>
            <span>Informations</span>
        </div>
        <div class="step" data-step="2">
            <div class="step-number">2</div>
            <span>Audience</span>
        </div>
        <div class="step" data-step="3">
            <div class="step-number">3</div>
            <span>Contenu</span>
        </div>
        <div class="step" data-step="4">
            <div class="step-number">4</div>
            <span>Révision</span>
        </div>
    </div>

    <form method="POST" id="campaignForm">
        <!-- Étape 1: Informations générales -->
        <div class="form-section active" data-section="1">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-info-circle me-2"></i>Informations de la campagne</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-8">
                            <div class="mb-3">
                                <label class="form-label">Nom de la campagne *</label>
                                <input type="text" class="form-control" name="campaign_name" required
                                       placeholder="Ex: Newsletter Décembre 2025">
                                <div class="form-text">Nom interne pour identifier votre campagne</div>
                            </div>
                            
                            <div class="mb-3">
                                <label class="form-label">Objet de l'email *</label>
                                <input type="text" class="form-control" name="subject" required
                                       placeholder="Ex: Nos dernières actualités et promotions">
                                <div class="form-text">Ce que verront vos destinataires dans leur boîte email</div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label">Nom expéditeur *</label>
                                        <input type="text" class="form-control" name="from_name" 
                                               value="iDkom" required>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label">Email expéditeur *</label>
                                        <input type="email" class="form-control" name="from_email" 
                                               value="contact@idkom.fr" required>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card bg-light">
                                <div class="card-body">
                                    <h6><i class="fas fa-lightbulb me-2"></i>Conseils</h6>
                                    <ul class="list-unstyled small">
                                        <li><i class="fas fa-check text-success me-1"></i>Choisissez un objet accrocheur</li>
                                        <li><i class="fas fa-check text-success me-1"></i>Évitez les mots "spam"</li>
                                        <li><i class="fas fa-check text-success me-1"></i>Limitez à 50 caractères</li>
                                        <li><i class="fas fa-check text-success me-1"></i>Personnalisez l'expéditeur</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Étape 2: Audience -->
        <div class="form-section" data-section="2">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-users me-2"></i>Sélection de l'audience</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-8">
                            <div class="mb-4">
                                <label class="form-label">Mode de ciblage</label>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="card target-option" data-target="all">
                                            <div class="card-body text-center">
                                                <i class="fas fa-globe fa-2x text-primary mb-2"></i>
                                                <h6>Tous les contacts</h6>
                                                <p class="small text-muted">Envoyer à tous vos contacts actifs</p>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="target_mode" 
                                                           value="all" id="target_all" checked>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="card target-option" data-target="segment">
                                            <div class="card-body text-center">
                                                <i class="fas fa-filter fa-2x text-success mb-2"></i>
                                                <h6>Segment personnalisé</h6>
                                                <p class="small text-muted">Cibler selon des critères précis</p>
                                                <div class="form-check">
                                                    <input class="form-check-input" type="radio" name="target_mode" 
                                                           value="segment" id="target_segment">
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Options de segment -->
                            <div id="segmentOptions" style="display: none;">
                                <div class="mb-3">
                                    <label class="form-label">Critères de ciblage</label>
                                    <div class="row">
                                        <div class="col-md-4">
                                            <select class="form-select" name="filter_consent">
                                                <option value="">Tout consentement</option>
                                                <option value="opt_in">Opt-in uniquement</option>
                                                <option value="b2b_cold">B2B Cold</option>
                                            </select>
                                            <small class="text-muted">Statut consentement</small>
                                        </div>
                                        <div class="col-md-4">
                                            <select class="form-select" name="filter_industry">
                                                <option value="">Tous secteurs</option>
                                                <option value="Événementiel">Événementiel</option>
                                                <option value="Communication">Communication</option>
                                                <option value="Design">Design</option>
                                                <option value="Tech">Tech</option>
                                            </select>
                                            <small class="text-muted">Secteur d'activité</small>
                                        </div>
                                        <div class="col-md-4">
                                            <select class="form-select" name="filter_source">
                                                <option value="">Toutes sources</option>
                                                <option value="Salon">Salons</option>
                                                <option value="Site web">Site web</option>
                                                <option value="Import">Import fichier</option>
                                            </select>
                                            <small class="text-muted">Source des contacts</small>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="audience-stats">
                                <h6><i class="fas fa-chart-pie me-2"></i>Statistiques audience</h6>
                                <div class="row text-center">
                                    <div class="col-6">
                                        <h4 class="text-primary"><?php echo number_format($contactsCount['total']); ?></h4>
                                        <small>Contacts total</small>
                                    </div>
                                    <div class="col-6">
                                        <h4 class="text-success" id="targetedCount"><?php echo number_format($contactsCount['total']); ?></h4>
                                        <small>Seront ciblés</small>
                                    </div>
                                </div>
                                <hr>
                                <div class="small">
                                    <div class="d-flex justify-content-between">
                                        <span>Opt-in:</span>
                                        <span><?php echo number_format($contactsCount['opt_in']); ?></span>
                                    </div>
                                    <div class="d-flex justify-content-between">
                                        <span>B2B Cold:</span>
                                        <span><?php echo number_format($contactsCount['b2b_cold']); ?></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Étape 3: Contenu -->
        <div class="form-section" data-section="3">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-edit me-2"></i>Contenu de la campagne</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6>Choisir un template</h6>
                            <div class="template-grid">
                                <?php if (empty($templates)): ?>
                                    <div class="template-preview selected" data-template="default">
                                        <div class="d-flex align-items-center">
                                            <i class="fas fa-file-alt fa-2x text-primary me-3"></i>
                                            <div>
                                                <h6>Template par défaut</h6>
                                                <small class="text-muted">Template simple et professionnel</small>
                                            </div>
                                        </div>
                                        <input type="radio" name="template_id" value="default" checked style="display: none;">
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($templates as $template): ?>
                                        <div class="template-preview" data-template="<?php echo $template['id']; ?>">
                                            <div class="d-flex align-items-center">
                                                <i class="fas fa-file-alt fa-2x text-primary me-3"></i>
                                                <div>
                                                    <h6><?php echo htmlspecialchars($template['name']); ?></h6>
                                                    <small class="text-muted"><?php echo htmlspecialchars($template['description']); ?></small>
                                                </div>
                                            </div>
                                            <input type="radio" name="template_id" value="<?php echo $template['id']; ?>" style="display: none;">
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>

                            <div class="mt-4">
                                <h6>Contenu personnalisé</h6>
                                <div class="mb-3">
                                    <label class="form-label">Message principal</label>
                                    <textarea class="form-control" name="custom_content" rows="8" 
                                              placeholder="Rédigez ici le contenu de votre newsletter..."></textarea>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Call-to-action (optionnel)</label>
                                    <div class="row">
                                        <div class="col-md-8">
                                            <input type="text" class="form-control" name="cta_text" 
                                                   placeholder="Ex: Découvrir nos services">
                                        </div>
                                        <div class="col-md-4">
                                            <input type="url" class="form-control" name="cta_url" 
                                                   placeholder="https://...">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <h6>Aperçu en temps réel</h6>
                            <div class="preview-container p-3" id="emailPreview">
                                <div class="email-header mb-3">
                                    <strong>De:</strong> <span id="previewFrom">iDkom &lt;contact@idkom.fr&gt;</span><br>
                                    <strong>Objet:</strong> <span id="previewSubject">Votre objet d'email</span>
                                </div>
                                <hr>
                                <div id="previewContent">
                                    <p>Votre contenu apparaîtra ici...</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Étape 4: Révision -->
        <div class="form-section" data-section="4">
            <div class="card">
                <div class="card-header">
                    <h5><i class="fas fa-check-circle me-2"></i>Révision et envoi</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-8">
                            <div class="card">
                                <div class="card-body">
                                    <h6>Résumé de la campagne</h6>
                                    <table class="table table-borderless">
                                        <tr>
                                            <td><strong>Nom:</strong></td>
                                            <td id="reviewName">-</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Objet:</strong></td>
                                            <td id="reviewSubject">-</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Expéditeur:</strong></td>
                                            <td id="reviewFrom">-</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Destinataires:</strong></td>
                                            <td id="reviewRecipients">-</td>
                                        </tr>
                                        <tr>
                                            <td><strong>Template:</strong></td>
                                            <td id="reviewTemplate">-</td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <div class="mt-4">
                                <h6>Options d'envoi</h6>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="send_mode" 
                                                   value="now" id="send_now" checked>
                                            <label class="form-check-label" for="send_now">
                                                <strong>Envoyer maintenant</strong><br>
                                                <small class="text-muted">Démarrer l'envoi immédiatement</small>
                                            </label>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-check">
                                            <input class="form-check-input" type="radio" name="send_mode" 
                                                   value="draft" id="send_draft">
                                            <label class="form-check-label" for="send_draft">
                                                <strong>Sauver comme brouillon</strong><br>
                                                <small class="text-muted">Finaliser plus tard</small>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card bg-warning text-dark">
                                <div class="card-body">
                                    <h6><i class="fas fa-exclamation-triangle me-2"></i>Attention</h6>
                                    <ul class="list-unstyled small mb-0">
                                        <li>• Vérifiez l'objet et le contenu</li>
                                        <li>• Contrôlez la liste des destinataires</li>
                                        <li>• L'envoi ne peut pas être annulé</li>
                                        <li>• Testez d'abord avec un brouillon</li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Boutons de navigation -->
        <div class="d-flex justify-content-between mt-4 mb-5">
            <button type="button" class="btn btn-outline-secondary" id="prevBtn" style="display: none;">
                <i class="fas fa-arrow-left me-1"></i>Précédent
            </button>
            <div></div>
            <div>
                <button type="button" class="btn btn-primary" id="nextBtn">
                    Suivant <i class="fas fa-arrow-right ms-1"></i>
                </button>
                <button type="submit" class="btn btn-success" id="submitBtn" style="display: none;">
                    <i class="fas fa-check me-1"></i>Créer la campagne
                </button>
            </div>
        </div>
    </form>
</div>

<script>
let currentStep = 1;
const maxStep = 4;

// Navigation entre étapes
document.getElementById('nextBtn').addEventListener('click', function() {
    if (validateCurrentStep()) {
        if (currentStep < maxStep) {
            currentStep++;
            showStep(currentStep);
        }
    }
});

document.getElementById('prevBtn').addEventListener('click', function() {
    if (currentStep > 1) {
        currentStep--;
        showStep(currentStep);
    }
});

function showStep(step) {
    // Masquer toutes les sections
    document.querySelectorAll('.form-section').forEach(section => {
        section.classList.remove('active');
    });
    
    // Afficher la section courante
    document.querySelector(`[data-section="${step}"]`).classList.add('active');
    
    // Mettre à jour les indicateurs
    document.querySelectorAll('.step').forEach(stepEl => {
        const stepNum = parseInt(stepEl.dataset.step);
        stepEl.classList.remove('active', 'completed');
        
        if (stepNum === step) {
            stepEl.classList.add('active');
        } else if (stepNum < step) {
            stepEl.classList.add('completed');
        }
    });
    
    // Gestion des boutons
    document.getElementById('prevBtn').style.display = step > 1 ? 'block' : 'none';
    document.getElementById('nextBtn').style.display = step < maxStep ? 'block' : 'none';
    document.getElementById('submitBtn').style.display = step === maxStep ? 'block' : 'none';
    
    // Actions spécifiques par étape
    if (step === 4) {
        updateReview();
    }
}

function validateCurrentStep() {
    switch (currentStep) {
        case 1:
            const name = document.querySelector('[name="campaign_name"]').value;
            const subject = document.querySelector('[name="subject"]').value;
            if (!name || !subject) {
                alert('Veuillez remplir tous les champs obligatoires');
                return false;
            }
            break;
        case 2:
            // Validation audience
            break;
        case 3:
            // Validation contenu
            break;
    }
    return true;
}

// Gestion des templates
document.querySelectorAll('.template-preview').forEach(preview => {
    preview.addEventListener('click', function() {
        document.querySelectorAll('.template-preview').forEach(p => p.classList.remove('selected'));
        this.classList.add('selected');
        this.querySelector('input[type="radio"]').checked = true;
    });
});

// Gestion du ciblage
document.querySelectorAll('[name="target_mode"]').forEach(radio => {
    radio.addEventListener('change', function() {
        const segmentOptions = document.getElementById('segmentOptions');
        if (this.value === 'segment') {
            segmentOptions.style.display = 'block';
        } else {
            segmentOptions.style.display = 'none';
        }
    });
});

// Aperçu en temps réel
function updatePreview() {
    const fromName = document.querySelector('[name="from_name"]').value || 'iDkom';
    const fromEmail = document.querySelector('[name="from_email"]').value || 'contact@idkom.fr';
    const subject = document.querySelector('[name="subject"]').value || 'Votre objet d\'email';
    const content = document.querySelector('[name="custom_content"]').value || 'Votre contenu apparaîtra ici...';
    
    document.getElementById('previewFrom').textContent = `${fromName} <${fromEmail}>`;
    document.getElementById('previewSubject').textContent = subject;
    document.getElementById('previewContent').innerHTML = content.replace(/\n/g, '<br>');
}

// Écouter les changements de contenu
document.querySelector('[name="from_name"]').addEventListener('input', updatePreview);
document.querySelector('[name="from_email"]').addEventListener('input', updatePreview);
document.querySelector('[name="subject"]').addEventListener('input', updatePreview);
document.querySelector('[name="custom_content"]').addEventListener('input', updatePreview);

// Mettre à jour la révision
function updateReview() {
    document.getElementById('reviewName').textContent = document.querySelector('[name="campaign_name"]').value;
    document.getElementById('reviewSubject').textContent = document.querySelector('[name="subject"]').value;
    document.getElementById('reviewFrom').textContent = `${document.querySelector('[name="from_name"]').value} <${document.querySelector('[name="from_email"]').value}>`;
    
    const targetMode = document.querySelector('[name="target_mode"]:checked').value;
    document.getElementById('reviewRecipients').textContent = targetMode === 'all' ? 'Tous les contacts' : 'Segment personnalisé';
    
    const template = document.querySelector('[name="template_id"]:checked');
    document.getElementById('reviewTemplate').textContent = template ? 'Template sélectionné' : 'Template par défaut';
}
</script>

<?php
/**
 * Créer une nouvelle campagne
 */
function createCampaign($db, $data) {
    try {
        $stmt = $db->prepare("
            INSERT INTO newsletter_campaigns 
            (name, subject, from_name, from_email, template_id, html_override, 
             target_mode, status, created_at, updated_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())
        ");
        
        $status = ($data['send_mode'] ?? 'draft') === 'now' ? 'scheduled' : 'draft';
        $templateId = $data['template_id'] === 'default' ? null : intval($data['template_id']);
        
        $result = $stmt->execute([
            $data['campaign_name'],
            $data['subject'],
            $data['from_name'],
            $data['from_email'],
            $templateId,
            $data['custom_content'] ?? null,
            $data['target_mode'] ?? 'all',
            $status
        ]);
        
        if ($result) {
            $campaignId = $db->lastInsertId();
            
            // Enregistrer les critères de ciblage si segment
            if (($data['target_mode'] ?? 'all') === 'segment') {
                saveCampaignTargeting($db, $campaignId, $data);
            }
            
            return [
                'success' => true,
                'campaign_id' => $campaignId
            ];
        }
        
        return [
            'success' => false,
            'message' => 'Erreur lors de la création'
        ];
        
    } catch (PDOException $e) {
        error_log("Erreur création campagne: " . $e->getMessage());
        return [
            'success' => false,
            'message' => 'Erreur base de données'
        ];
    }
}

/**
 * Récupérer les templates disponibles
 */
function getTemplates($db) {
    try {
        $stmt = $db->query("SELECT * FROM newsletter_templates ORDER BY created_at DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Erreur récupération templates: " . $e->getMessage());
        return [];
    }
}

/**
 * Récupérer les segments disponibles
 */
function getSegments($db) {
    try {
        $stmt = $db->query("SELECT * FROM newsletter_segments ORDER BY created_at DESC");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Erreur récupération segments: " . $e->getMessage());
        return [];
    }
}

/**
 * Compter les contacts par catégorie
 */
function getContactsCount($db) {
    $counts = [
        'total' => 0,
        'opt_in' => 0,
        'b2b_cold' => 0,
        'opt_out' => 0
    ];
    
    try {
        $stmt = $db->query("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN consent_status = 'opt_in' THEN 1 ELSE 0 END) as opt_in,
                SUM(CASE WHEN consent_status = 'b2b_cold' THEN 1 ELSE 0 END) as b2b_cold,
                SUM(CASE WHEN consent_status = 'opt_out' THEN 1 ELSE 0 END) as opt_out
            FROM newsletter_contacts
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return array_merge($counts, $result);
    } catch (PDOException $e) {
        error_log("Erreur comptage contacts: " . $e->getMessage());
        return $counts;
    }
}

/**
 * Sauvegarder les critères de ciblage
 */
function saveCampaignTargeting($db, $campaignId, $data) {
    // Ici on pourrait créer un segment dynamique basé sur les critères
    // Pour l'instant, on stocke les critères dans les notes de la campagne
    
    $criteria = [
        'consent' => $data['filter_consent'] ?? '',
        'industry' => $data['filter_industry'] ?? '',
        'source' => $data['filter_source'] ?? ''
    ];
    
    try {
        $stmt = $db->prepare("UPDATE newsletter_campaigns SET notes = ? WHERE id = ?");
        $stmt->execute([json_encode($criteria), $campaignId]);
    } catch (PDOException $e) {
        error_log("Erreur sauvegarde ciblage: " . $e->getMessage());
    }
}

require_once dirname(__FILE__) . '/../../../includes/footer.php';
?>