<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once dirname(__FILE__) . '/../../../includes/config.php';
require_once dirname(__FILE__) . '/../../../includes/auth.php';

$page_title = 'Gestion des Campagnes Newsletter';
requireAdminAuth();

// Récupération des campagnes
$campaigns = getCampaigns($db);
$stats = getCampaignsStats($db);

require_once dirname(__FILE__) . '/../../../includes/header.php';
?>

<style>
.campaigns-header {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%);
    color: white;
    padding: 2rem 0;
    margin-bottom: 2rem;
}

.campaign-card {
    transition: all 0.3s ease;
    border: 1px solid #dee2e6;
}

.campaign-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
}

.status-badge {
    font-size: 0.75rem;
    padding: 6px 12px;
    border-radius: 20px;
}

.stats-card {
    background: white;
    border-radius: 10px;
    padding: 1.5rem;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.quick-actions {
    background: #f8f9fa;
    border-radius: 10px;
    padding: 1.5rem;
}
</style>

<div class="campaigns-header">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h1><i class="fas fa-paper-plane me-3"></i>Gestion des Campagnes</h1>
                <p class="mb-0">Créez et gérez vos campagnes d'emailing</p>
            </div>
            <div class="col-md-4 text-end">
                <a href="create.php" class="btn btn-light btn-lg">
                    <i class="fas fa-plus me-2"></i>Nouvelle Campagne
                </a>
            </div>
        </div>
    </div>
</div>

<div class="container">
    <!-- Statistiques globales -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="stats-card text-center">
                <h3 class="text-primary"><?php echo number_format($stats['total_campaigns']); ?></h3>
                <small class="text-muted">Campagnes créées</small>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stats-card text-center">
                <h3 class="text-success"><?php echo number_format($stats['emails_sent']); ?></h3>
                <small class="text-muted">Emails envoyés</small>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stats-card text-center">
                <h3 class="text-info"><?php echo $stats['avg_open_rate']; ?>%</h3>
                <small class="text-muted">Taux d'ouverture</small>
            </div>
        </div>
        <div class="col-md-3">
            <div class="stats-card text-center">
                <h3 class="text-warning"><?php echo $stats['avg_click_rate']; ?>%</h3>
                <small class="text-muted">Taux de clic</small>
            </div>
        </div>
    </div>

    <!-- Actions rapides -->
    <div class="row mb-4">
        <div class="col-md-8">
            <div class="quick-actions">
                <h5><i class="fas fa-bolt me-2"></i>Actions rapides</h5>
                <div class="row">
                    <div class="col-md-4">
                        <a href="create.php" class="btn btn-outline-primary w-100 mb-2">
                            <i class="fas fa-plus me-1"></i>Nouvelle campagne
                        </a>
                    </div>
                    <div class="col-md-4">
                        <a href="../templates/index.php" class="btn btn-outline-info w-100 mb-2">
                            <i class="fas fa-palette me-1"></i>Gérer templates
                        </a>
                    </div>
                    <div class="col-md-4">
                        <a href="../lists/contacts.php" class="btn btn-outline-success w-100 mb-2">
                            <i class="fas fa-users me-1"></i>Voir contacts
                        </a>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-md-4">
            <div class="card">
                <div class="card-header">
                    <h6 class="mb-0"><i class="fas fa-clock me-2"></i>Dernière activité</h6>
                </div>
                <div class="card-body">
                    <small class="text-muted">
                        <?php if (!empty($campaigns)): ?>
                            Dernière campagne : <?php echo date('d/m/Y H:i', strtotime($campaigns[0]['created_at'])); ?>
                        <?php else: ?>
                            Aucune campagne créée
                        <?php endif; ?>
                    </small>
                </div>
            </div>
        </div>
    </div>

    <!-- Liste des campagnes -->
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">
                        <i class="fas fa-list me-2"></i>Vos Campagnes 
                        (<?php echo count($campaigns); ?>)
                    </h5>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-secondary active" data-filter="all">Toutes</button>
                        <button class="btn btn-outline-secondary" data-filter="draft">Brouillons</button>
                        <button class="btn btn-outline-secondary" data-filter="sent">Envoyées</button>
                        <button class="btn btn-outline-secondary" data-filter="scheduled">Programmées</button>
                    </div>
                </div>
                <div class="card-body">
                    <?php if (empty($campaigns)): ?>
                        <!-- État vide -->
                        <div class="text-center py-5">
                            <i class="fas fa-paper-plane text-muted" style="font-size: 4rem;"></i>
                            <h4 class="mt-3">Aucune campagne</h4>
                            <p class="text-muted">Créez votre première campagne newsletter pour commencer à communiquer avec vos contacts.</p>
                            <a href="create.php" class="btn btn-primary btn-lg">
                                <i class="fas fa-plus me-2"></i>Créer ma première campagne
                            </a>
                        </div>
                    <?php else: ?>
                        <!-- Grille des campagnes -->
                        <div class="row" id="campaignsGrid">
                            <?php foreach ($campaigns as $campaign): ?>
                                <div class="col-md-6 col-lg-4 mb-4 campaign-item" data-status="<?php echo $campaign['status']; ?>">
                                    <div class="campaign-card card h-100">
                                        <div class="card-header d-flex justify-content-between align-items-center">
                                            <span class="badge status-badge <?php echo getStatusClass($campaign['status']); ?>">
                                                <?php echo getStatusLabel($campaign['status']); ?>
                                            </span>
                                            <div class="dropdown">
                                                <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="dropdown">
                                                    <i class="fas fa-ellipsis-v"></i>
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li><a class="dropdown-item" href="edit.php?id=<?php echo $campaign['id']; ?>">
                                                        <i class="fas fa-edit me-2"></i>Modifier
                                                    </a></li>
                                                    <?php if ($campaign['status'] === 'draft'): ?>
                                                        <li><a class="dropdown-item" href="send.php?id=<?php echo $campaign['id']; ?>">
                                                            <i class="fas fa-paper-plane me-2"></i>Envoyer
                                                        </a></li>
                                                    <?php endif; ?>
                                                    <li><a class="dropdown-item" href="stats.php?id=<?php echo $campaign['id']; ?>">
                                                        <i class="fas fa-chart-bar me-2"></i>Statistiques
                                                    </a></li>
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li><a class="dropdown-item text-danger" href="#" onclick="deleteCampaign(<?php echo $campaign['id']; ?>)">
                                                        <i class="fas fa-trash me-2"></i>Supprimer
                                                    </a></li>
                                                </ul>
                                            </div>
                                        </div>
                                        <div class="card-body">
                                            <h6 class="card-title"><?php echo htmlspecialchars($campaign['name']); ?></h6>
                                            <p class="card-text text-muted small">
                                                <?php echo htmlspecialchars($campaign['subject']); ?>
                                            </p>
                                            
                                            <!-- Statistiques de la campagne -->
                                            <?php if ($campaign['status'] !== 'draft'): ?>
                                                <div class="row text-center mt-3">
                                                    <div class="col-4">
                                                        <div class="small text-muted">Envoyés</div>
                                                        <strong><?php echo number_format($campaign['emails_sent'] ?? 0); ?></strong>
                                                    </div>
                                                    <div class="col-4">
                                                        <div class="small text-muted">Ouvertures</div>
                                                        <strong><?php echo $campaign['open_rate'] ?? 0; ?>%</strong>
                                                    </div>
                                                    <div class="col-4">
                                                        <div class="small text-muted">Clics</div>
                                                        <strong><?php echo $campaign['click_rate'] ?? 0; ?>%</strong>
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                                <div class="text-center mt-3">
                                                    <span class="text-muted small">Brouillon - Prêt à envoyer</span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="card-footer">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <small class="text-muted">
                                                    <i class="fas fa-calendar me-1"></i>
                                                    <?php echo date('d/m/Y', strtotime($campaign['created_at'])); ?>
                                                </small>
                                                <div class="btn-group btn-group-sm">
                                                    <a href="edit.php?id=<?php echo $campaign['id']; ?>" class="btn btn-outline-primary">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <a href="stats.php?id=<?php echo $campaign['id']; ?>" class="btn btn-outline-info">
                                                        <i class="fas fa-chart-bar"></i>
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Filtrage des campagnes
document.querySelectorAll('[data-filter]').forEach(btn => {
    btn.addEventListener('click', function() {
        const filter = this.dataset.filter;
        
        // Mise à jour des boutons actifs
        document.querySelectorAll('[data-filter]').forEach(b => b.classList.remove('active'));
        this.classList.add('active');
        
        // Filtrage des campagnes
        document.querySelectorAll('.campaign-item').forEach(item => {
            if (filter === 'all' || item.dataset.status === filter) {
                item.style.display = 'block';
            } else {
                item.style.display = 'none';
            }
        });
    });
});

// Suppression de campagne
function deleteCampaign(id) {
    if (confirm('Êtes-vous sûr de vouloir supprimer cette campagne ? Cette action est irréversible.')) {
        // TODO: Implémenter la suppression via AJAX
        fetch(`delete.php?id=${id}`, {
            method: 'POST'
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            } else {
                alert('Erreur lors de la suppression');
            }
        })
        .catch(error => {
            console.error('Erreur:', error);
            alert('Erreur de communication avec le serveur');
        });
    }
}
</script>

<?php
/**
 * Récupérer toutes les campagnes
 */
function getCampaigns($db) {
    try {
        $stmt = $db->query("
            SELECT c.*, 
                   COUNT(d.id) as emails_sent,
                   ROUND(AVG(CASE WHEN d.first_open_at IS NOT NULL THEN 1 ELSE 0 END) * 100, 1) as open_rate,
                   ROUND(AVG(CASE WHEN d.first_click_at IS NOT NULL THEN 1 ELSE 0 END) * 100, 1) as click_rate
            FROM newsletter_campaigns c
            LEFT JOIN newsletter_deliveries d ON c.id = d.campaign_id
            GROUP BY c.id
            ORDER BY c.created_at DESC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Erreur récupération campagnes: " . $e->getMessage());
        return [];
    }
}

/**
 * Récupérer les statistiques globales
 */
function getCampaignsStats($db) {
    $stats = [
        'total_campaigns' => 0,
        'emails_sent' => 0,
        'avg_open_rate' => 0,
        'avg_click_rate' => 0
    ];
    
    try {
        // Nombre total de campagnes
        $stmt = $db->query("SELECT COUNT(*) FROM newsletter_campaigns");
        $stats['total_campaigns'] = $stmt->fetchColumn();
        
        // Emails envoyés
        $stmt = $db->query("SELECT COUNT(*) FROM newsletter_deliveries WHERE status = 'sent'");
        $stats['emails_sent'] = $stmt->fetchColumn();
        
        // Taux d'ouverture moyen
        $stmt = $db->query("
            SELECT ROUND(AVG(CASE WHEN first_open_at IS NOT NULL THEN 1 ELSE 0 END) * 100, 1)
            FROM newsletter_deliveries 
            WHERE status = 'sent'
        ");
        $stats['avg_open_rate'] = $stmt->fetchColumn() ?: 0;
        
        // Taux de clic moyen
        $stmt = $db->query("
            SELECT ROUND(AVG(CASE WHEN first_click_at IS NOT NULL THEN 1 ELSE 0 END) * 100, 1)
            FROM newsletter_deliveries 
            WHERE status = 'sent'
        ");
        $stats['avg_click_rate'] = $stmt->fetchColumn() ?: 0;
        
    } catch (PDOException $e) {
        error_log("Erreur stats campagnes: " . $e->getMessage());
    }
    
    return $stats;
}

/**
 * Obtenir la classe CSS pour le statut
 */
function getStatusClass($status) {
    $classes = [
        'draft' => 'bg-secondary',
        'scheduled' => 'bg-warning text-dark',
        'sending' => 'bg-primary',
        'finished' => 'bg-success',
        'paused' => 'bg-info'
    ];
    return $classes[$status] ?? 'bg-secondary';
}

/**
 * Obtenir le label pour le statut
 */
function getStatusLabel($status) {
    $labels = [
        'draft' => 'Brouillon',
        'scheduled' => 'Programmée',
        'sending' => 'En cours',
        'finished' => 'Envoyée',
        'paused' => 'En pause'
    ];
    return $labels[$status] ?? ucfirst($status);
}

require_once dirname(__FILE__) . '/../../../includes/footer.php';
?>