<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once dirname(__FILE__) . '/../../../includes/config.php';
require_once dirname(__FILE__) . '/../../../includes/auth.php';

$page_title = 'Import Newsletter';
requireAdminAuth(); // Utilise la fonction d'auth admin

require_once dirname(__FILE__) . '/../../../includes/header.php';
?>

<style>
.upload-zone {
    border: 3px dashed #6c757d;
    border-radius: 10px;
    padding: 3rem;
    text-align: center;
    background: white;
    transition: all 0.3s ease;
    cursor: pointer;
}

.upload-zone:hover {
    background: #f8f9fa;
    border-color: #0d6efd;
}

.upload-zone.dragover {
    background: #e3f2fd;
    border-color: #198754;
}

.upload-icon {
    font-size: 4rem;
    color: #6c757d;
    margin-bottom: 1rem;
}

.upload-zone:hover .upload-icon {
    color: #0d6efd;
}

.progress-section {
    display: none;
}

.file-preview {
    display: none;
}

.table-preview {
    max-height: 300px;
    overflow-y: auto;
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
}

.column-mapping {
    background: #f8f9fa;
    padding: 1rem;
    border-radius: 0.375rem;
    margin-top: 1rem;
}

.mapping-item {
    display: flex;
    align-items: center;
    margin-bottom: 1rem;
}

.mapping-item:last-child {
    margin-bottom: 0;
}

.detection-badge {
    font-size: 0.75rem;
    padding: 4px 8px;
}
</style>

<div class="container mt-5">
    <h1 class="mb-4"><i class="fas fa-envelope-open-text me-2"></i>Import Newsletter</h1>

    <!-- Zone d'upload -->
    <div class="row">
        <div class="col-lg-8">
            <div class="card">
                <div class="card-body">
                    <div class="upload-zone" id="uploadZone">
                        <input type="file" id="fileInput" accept=".csv,.xlsx,.xls" style="display: none;">
                        <div class="upload-icon">
                            <i class="fas fa-cloud-upload-alt"></i>
                        </div>
                        <h4>Glissez-déposez votre fichier ici</h4>
                        <p class="text-muted">ou cliquez pour sélectionner</p>
                        <div class="mt-3">
                            <span class="badge bg-secondary me-2">CSV</span>
                            <span class="badge bg-secondary me-2">Excel (.xlsx)</span>
                            <span class="badge bg-secondary">Excel (.xls)</span>
                        </div>
                        <small class="text-muted d-block mt-2">Taille maximum : 50 MB</small>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-4">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-info-circle me-2"></i>Conseils d'import</h5>
                </div>
                <div class="card-body">
                    <ul class="list-unstyled">
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Inclure au minimum une colonne "Email"</li>
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Ajouter Nom, Prénom si disponible</li>
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Entreprise et Secteur recommandés</li>
                        <li class="mb-2"><i class="fas fa-check text-success me-2"></i>Éviter les caractères spéciaux</li>
                    </ul>
                    <hr>
                    <h6><i class="fas fa-file-alt me-2"></i>Formats acceptés :</h6>
                    <small class="text-muted">
                        • CSV avec séparateur virgule ou point-virgule<br>
                        • Excel moderne (.xlsx)<br>
                        • Excel legacy (.xls)
                    </small>
                </div>
            </div>
        </div>
    </div>

    <!-- Section de progression -->
    <div class="progress-section mt-4" id="progressSection">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-cog fa-spin me-2"></i>Analyse en cours...</h5>
            </div>
            <div class="card-body">
                <div class="progress mb-3" style="height: 20px;">
                    <div class="progress-bar progress-bar-striped progress-bar-animated" 
                         role="progressbar" style="width: 0%" id="progressBar">0%</div>
                </div>
                <div class="row">
                    <div class="col-md-3 mb-3">
                        <div class="card">
                            <div class="card-body text-center">
                                <h2 class="text-primary" id="totalEmails">-</h2>
                                <p class="text-muted mb-0">Emails trouvés</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card">
                            <div class="card-body text-center">
                                <h2 class="text-success" id="validFormat">-</h2>
                                <p class="text-muted mb-0">Format valide</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card">
                            <div class="card-body text-center">
                                <h2 class="text-warning" id="duplicates">-</h2>
                                <p class="text-muted mb-0">Doublons</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3 mb-3">
                        <div class="card">
                            <div class="card-body text-center">
                                <h2 class="text-danger" id="suspicious">-</h2>
                                <p class="text-muted mb-0">Suspects</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Prévisualisation -->
    <div class="file-preview mt-4" id="filePreview">
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0"><i class="fas fa-table me-2"></i>Prévisualisation des données</h5>
                <span class="badge bg-info">Premières 10 lignes</span>
            </div>
            <div class="card-body">
                <div class="table-preview">
                    <table class="table table-striped table-sm mb-0" id="previewTable">
                        <thead class="table-dark">
                            <tr id="previewHeaders"></tr>
                        </thead>
                        <tbody id="previewBody"></tbody>
                    </table>
                </div>

                <!-- Mapping des colonnes -->
                <div class="column-mapping">
                    <h6><i class="fas fa-link me-2"></i>Détection automatique des colonnes</h6>
                    <div class="row" id="columnMappings">
                        <!-- Les mappings seront générés dynamiquement -->
                    </div>
                </div>

                <div class="text-end mt-4">
                    <button class="btn btn-primary me-2" id="validateBtn">
                        <i class="fas fa-check-circle me-1"></i>Valider et Nettoyer
                    </button>
                    <button class="btn btn-outline-secondary" id="cancelBtn">
                        <i class="fas fa-times me-1"></i>Annuler
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Alerte en cas d'erreur -->
    <div id="alertContainer"></div>
</div>

<script>
// Variables globales
let currentFile = null;
let fileData = null;

// Éléments DOM
const uploadZone = document.getElementById('uploadZone');
const fileInput = document.getElementById('fileInput');
const progressSection = document.getElementById('progressSection');
const filePreview = document.getElementById('filePreview');
const progressBar = document.getElementById('progressBar');
const alertContainer = document.getElementById('alertContainer');

// Gestion du drag & drop
uploadZone.addEventListener('click', () => fileInput.click());
uploadZone.addEventListener('dragover', handleDragOver);
uploadZone.addEventListener('dragleave', handleDragLeave);
uploadZone.addEventListener('drop', handleDrop);
fileInput.addEventListener('change', handleFileSelect);

function handleDragOver(e) {
    e.preventDefault();
    uploadZone.classList.add('dragover');
}

function handleDragLeave(e) {
    e.preventDefault();
    uploadZone.classList.remove('dragover');
}

function handleDrop(e) {
    e.preventDefault();
    uploadZone.classList.remove('dragover');
    const files = e.dataTransfer.files;
    if (files.length > 0) {
        processFile(files[0]);
    }
}

function handleFileSelect(e) {
    if (e.target.files.length > 0) {
        processFile(e.target.files[0]);
    }
}

function showAlert(message, type = 'danger') {
    const alertHtml = `
        <div class="alert alert-${type} alert-dismissible fade show mt-3" role="alert">
            <i class="fas fa-exclamation-triangle me-2"></i>${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    `;
    alertContainer.innerHTML = alertHtml;
}

function processFile(file) {
    // Validation du fichier
    if (!validateFile(file)) return;

    currentFile = file;
    showProgress();
    
    // Envoi du fichier vers le serveur pour analyse
    uploadFileToServer(file);
}

function validateFile(file) {
    const allowedTypes = ['text/csv', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'];
    const maxSize = 50 * 1024 * 1024; // 50MB

    if (!allowedTypes.some(type => file.type === type || file.name.toLowerCase().endsWith('.csv') || file.name.toLowerCase().endsWith('.xlsx') || file.name.toLowerCase().endsWith('.xls'))) {
        showAlert('Format de fichier non supporté. Utilisez CSV ou Excel.');
        return false;
    }

    if (file.size > maxSize) {
        showAlert('Fichier trop volumineux. Maximum 50MB.');
        return false;
    }

    return true;
}

function showProgress() {
    progressSection.style.display = 'block';
    filePreview.style.display = 'none';
    alertContainer.innerHTML = '';
}

function uploadFileToServer(file) {
    const formData = new FormData();
    formData.append('file', file);
    
    // Afficher la progression
    updateProgressStats('Analyse du fichier...', 25);
    
    fetch('process.php', {
        method: 'POST',
        body: formData
    })
    .then(response => {
        console.log('Response status:', response.status);
        console.log('Response headers:', response.headers);
        
        // Lire la réponse en texte d'abord pour debug
        return response.text();
    })
    .then(text => {
        console.log('Response text:', text);
        
        // Essayer de parser en JSON
        try {
            const data = JSON.parse(text);
            if (data.success) {
                // Mettre à jour les statistiques
                updateProgressStats('Analyse terminée', 100, data.stats);
                
                // Afficher les résultats après un délai
                setTimeout(() => {
                    showResults(data);
                }, 1000);
            } else {
                showAlert(data.message || 'Erreur lors de l\'analyse du fichier.');
            }
        } catch (jsonError) {
            console.error('Erreur parsing JSON:', jsonError);
            console.log('Réponse brute du serveur:', text);
            showAlert('Réponse invalide du serveur. Vérifiez la console pour plus de détails.');
        }
    })
    .catch(error => {
        console.error('Erreur fetch:', error);
        showAlert('Erreur de communication avec le serveur: ' + error.message);
    });
}

function updateProgressStats(message, progress, stats = null) {
    // Mettre à jour le message
    const header = document.querySelector('#progressSection .card-header h5');
    if (header) {
        header.innerHTML = `<i class="fas fa-cog fa-spin me-2"></i>${message}`;
    }
    
    // Mettre à jour la barre de progression
    progressBar.style.width = progress + '%';
    progressBar.textContent = Math.round(progress) + '%';
    
    // Mettre à jour les stats si disponibles
    if (stats) {
        document.getElementById('totalEmails').textContent = stats.total || '0';
        document.getElementById('validFormat').textContent = stats.valid || '0';
        document.getElementById('duplicates').textContent = stats.duplicates || '0';
        document.getElementById('suspicious').textContent = stats.suspicious || '0';
    }
}

function showResults(data) {
    progressSection.style.display = 'none';
    filePreview.style.display = 'block';
    
    if (data.preview) {
        generatePreviewTable(data.preview);
    }
    if (data.mappings) {
        setupColumnMappings(data.mappings);
    }
    
    // Stocker l'import_id pour la validation
    window.currentImportId = data.import_id;
}

function generatePreviewTable(preview) {
    if (!preview.headers || !preview.data) return;
    
    // Génération des headers
    const headerRow = document.getElementById('previewHeaders');
    headerRow.innerHTML = preview.headers.map(h => `<th>${h}</th>`).join('');

    // Génération du body
    const tbody = document.getElementById('previewBody');
    tbody.innerHTML = preview.data.map(row => 
        `<tr>${row.map(cell => `<td>${cell || '<span class="text-muted">-</span>'}</td>`).join('')}</tr>`
    ).join('');
}

function setupColumnMappings(mappings) {
    const container = document.getElementById('columnMappings');
    container.innerHTML = mappings.map(mapping => `
        <div class="col-md-6 mb-3">
            <div class="mapping-item">
                <div class="me-3" style="min-width: 120px;">
                    <strong>${mapping.detected}</strong>
                    <span class="badge detection-badge ${mapping.confidence >= 90 ? 'bg-success' : 'bg-warning'}">
                        ${mapping.confidence}%
                    </span>
                </div>
                <select class="form-select form-select-sm" name="mapping[${mapping.column}]">
                    <option value="${mapping.type}" selected>${getTypeLabel(mapping.type)}</option>
                    <option value="ignore">Ignorer</option>
                    <option value="email">Email</option>
                    <option value="first_name">Prénom</option>
                    <option value="last_name">Nom</option>
                    <option value="company">Entreprise</option>
                    <option value="industry">Secteur</option>
                    <option value="phone">Téléphone</option>
                </select>
            </div>
        </div>
    `).join('');
}

function getTypeLabel(type) {
    const labels = {
        'email': 'Email',
        'first_name': 'Prénom',
        'last_name': 'Nom', 
        'company': 'Entreprise',
        'industry': 'Secteur',
        'phone': 'Téléphone'
    };
    return labels[type] || type;
}

// Bouton de validation
document.getElementById('validateBtn').addEventListener('click', function() {
    if (!currentFile) {
        showAlert('Aucun fichier sélectionné.');
        return;
    }
    
    // Récupération des mappings
    const mappings = {};
    document.querySelectorAll('select[name^="mapping"]').forEach(select => {
        const column = select.name.match(/\[(.*)\]/)[1];
        mappings[column] = select.value;
    });
    
    // Redirection vers l'analyse avec les paramètres
    const params = new URLSearchParams({
        file: currentFile.name,
        mappings: JSON.stringify(mappings)
    });
    
    window.location.href = `analyze.php?${params.toString()}`;
});

// Bouton d'annulation
document.getElementById('cancelBtn').addEventListener('click', function() {
    location.reload();
});
</script>

<?php require_once dirname(__FILE__) . '/../../../includes/footer.php'; ?>