<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once dirname(__FILE__) . '/../../includes/config.php';
require_once dirname(__FILE__) . '/../../includes/auth.php';

$page_title = 'Newsletter - Dashboard';
requireAdminAuth(); // Utilise la fonction d'auth admin

require_once dirname(__FILE__) . '/../../includes/header.php';

// Récupération des statistiques Newsletter
$stats = [
    'contacts_total' => 0,
    'listes_actives' => 0,
    'campagnes_envoyees' => 0,
    'taux_ouverture' => 0
];

try {
    // Contacts total dans la base
    $stmt = $db->query("SELECT COUNT(*) FROM newsletter_contacts WHERE consent_status IN ('b2b_cold', 'opt_in')");
    $stats['contacts_total'] = $stmt->fetchColumn() ?: 0;
    
    // Segments actifs (simulé car pas encore de vrais segments)
    $stmt = $db->query("SELECT COUNT(*) FROM newsletter_segments");
    $stats['listes_actives'] = $stmt->fetchColumn() ?: 0;
    
    // Campagnes envoyées ce mois
    $stmt = $db->query("SELECT COUNT(*) FROM newsletter_campaigns WHERE status = 'finished' AND YEAR(finished_at) = YEAR(NOW()) AND MONTH(finished_at) = MONTH(NOW())");
    $stats['campagnes_envoyees'] = $stmt->fetchColumn() ?: 0;
    
    // Taux d'ouverture moyen (calculé sur les deliveries)
    $stmt = $db->query("
        SELECT ROUND(
            (COUNT(CASE WHEN first_open_at IS NOT NULL THEN 1 END) * 100.0) / 
            NULLIF(COUNT(*), 0), 1
        ) as taux_ouverture
        FROM newsletter_deliveries 
        WHERE status = 'delivered' 
        AND sent_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    ");
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    $stats['taux_ouverture'] = $result['taux_ouverture'] ?: 0;

} catch(PDOException $e) {
    // En cas d'erreur, garder les valeurs par défaut à 0
    error_log("Erreur statistiques newsletter: " . $e->getMessage());
}
?>

<div class="container mt-5">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1><i class="fas fa-envelope-open-text me-2"></i>Newsletter - Dashboard</h1>
        <div>
            <a href="campaigns/create.php" class="btn btn-primary">
                <i class="fas fa-plus me-1"></i>Nouvelle campagne
            </a>
        </div>
    </div>
    
    <!-- Première ligne : Cartes de statistiques -->
    <div class="row">
        <!-- Carte Contacts -->
        <div class="col-md-3 mb-4">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-users me-2"></i>Contacts
                    </h5>
                    <h2 class="text-primary"><?php echo number_format($stats['contacts_total'], 0, ',', ' '); ?></h2>
                    <p class="text-muted">contacts dans la base</p>
                    <a href="lists/contacts.php" class="btn btn-primary">Gérer les contacts</a>
                </div>
            </div>
        </div>

        <!-- Carte Listes -->
        <div class="col-md-3 mb-4">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-list me-2"></i>Listes
                    </h5>
                    <h2 class="text-success"><?php echo $stats['listes_actives']; ?></h2>
                    <p class="text-muted">listes actives</p>
                    <a href="lists/index.php" class="btn btn-success">Voir les listes</a>
                </div>
            </div>
        </div>

        <!-- Carte Campagnes -->
        <div class="col-md-3 mb-4">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-paper-plane me-2"></i>Campagnes
                    </h5>
                    <h2 class="text-info"><?php echo $stats['campagnes_envoyees']; ?></h2>
                    <p class="text-muted">envoyées ce mois</p>
                    <a href="campaigns/index.php" class="btn btn-info">Voir les campagnes</a>
                </div>
            </div>
        </div>

        <!-- Carte Performance -->
        <div class="col-md-3 mb-4">
            <div class="card">
                <div class="card-body">
                    <h5 class="card-title">
                        <i class="fas fa-chart-line me-2"></i>Performance
                    </h5>
                    <h2 class="text-warning"><?php echo $stats['taux_ouverture']; ?>%</h2>
                    <p class="text-muted">taux d'ouverture moyen</p>
                    <a href="stats/index.php" class="btn btn-warning">Voir les stats</a>
                </div>
            </div>
        </div>
    </div>

    <!-- Deuxième ligne : Actions principales -->
    <div class="row">
        <div class="col-md-4 mb-4">
            <div class="card h-100">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">
                        <i class="fas fa-file-import me-2"></i>Import de contacts
                    </h5>
                </div>
                <div class="card-body">
                    <p>Importez vos listes d'emails depuis un fichier CSV ou Excel avec validation automatique.</p>
                    <ul class="list-unstyled">
                        <li><i class="fas fa-check text-success me-2"></i>Validation des emails</li>
                        <li><i class="fas fa-check text-success me-2"></i>Détection des doublons</li>
                        <li><i class="fas fa-check text-success me-2"></i>Correction automatique</li>
                        <li><i class="fas fa-check text-success me-2"></i>Nettoyage des données</li>
                    </ul>
                    <a href="import/upload.php" class="btn btn-primary">
                        <i class="fas fa-upload me-1"></i>Importer des contacts
                    </a>
                </div>
            </div>
        </div>

        <div class="col-md-4 mb-4">
            <div class="card h-100">
                <div class="card-header bg-success text-white">
                    <h5 class="mb-0">
                        <i class="fas fa-edit me-2"></i>Créer une campagne
                    </h5>
                </div>
                <div class="card-body">
                    <p>Créez et envoyez des newsletters professionnelles à vos contacts et prospects.</p>
                    <ul class="list-unstyled">
                        <li><i class="fas fa-check text-success me-2"></i>Templates responsive</li>
                        <li><i class="fas fa-check text-success me-2"></i>Éditeur visuel</li>
                        <li><i class="fas fa-check text-success me-2"></i>Personnalisation avancée</li>
                        <li><i class="fas fa-check text-success me-2"></i>Programmation d'envoi</li>
                    </ul>
                    <a href="campaigns/create.php" class="btn btn-success">
                        <i class="fas fa-plus me-1"></i>Nouvelle campagne
                    </a>
                </div>
            </div>
        </div>

        <div class="col-md-4 mb-4">
            <div class="card h-100">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0">
                        <i class="fas fa-chart-bar me-2"></i>Statistiques
                    </h5>
                </div>
                <div class="card-body">
                    <p>Analysez les performances de vos campagnes et optimisez vos envois.</p>
                    <ul class="list-unstyled">
                        <li><i class="fas fa-check text-success me-2"></i>Taux d'ouverture</li>
                        <li><i class="fas fa-check text-success me-2"></i>Clics et conversions</li>
                        <li><i class="fas fa-check text-success me-2"></i>Désabonnements</li>
                        <li><i class="fas fa-check text-success me-2"></i>Rapports détaillés</li>
                    </ul>
                    <a href="stats/index.php" class="btn btn-info">
                        <i class="fas fa-chart-line me-1"></i>Voir les statistiques
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Troisième ligne : Dernières activités -->
    <div class="card mt-4">
        <div class="card-header">
            <h5 class="mb-0"><i class="fas fa-history me-2"></i>Dernières activités</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Action</th>
                            <th>Détails</th>
                            <th>Utilisateur</th>
                            <th>Statut</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        try {
                            // Récupérer les dernières activités newsletter
                            $query = "
                                SELECT 
                                    'import' as type,
                                    il.filename as details,
                                    il.imported_by as user,
                                    il.imported_at as date,
                                    CONCAT(il.counts_clean, ' contacts importés') as status
                                FROM newsletter_import_logs il
                                ORDER BY il.imported_at DESC
                                LIMIT 5
                            ";
                            
                            $stmt = $db->query($query);
                            $activities = $stmt->fetchAll(PDO::FETCH_ASSOC);

                            if (empty($activities)) {
                                // Si pas d'activités, afficher un message
                                echo '<tr><td colspan="5" class="text-center py-4">';
                                echo '<i class="fas fa-info-circle text-muted me-2"></i>';
                                echo 'Aucune activité newsletter pour le moment. Commencez par importer des contacts !';
                                echo '</td></tr>';
                            } else {
                                foreach($activities as $activity) {
                                    $icon = 'fas fa-file-import text-primary';
                                    $badge_class = 'bg-primary';
                                    $action_text = 'Import contacts';
                                    
                                    echo '<tr>';
                                    echo '<td>' . date('d/m/Y H:i', strtotime($activity['date'])) . '</td>';
                                    echo '<td><i class="' . $icon . ' me-1"></i>' . $action_text . '</td>';
                                    echo '<td>' . htmlspecialchars($activity['details']) . '</td>';
                                    echo '<td>' . htmlspecialchars($activity['user']) . '</td>';
                                    echo '<td><span class="badge ' . $badge_class . '">' . htmlspecialchars($activity['status']) . '</span></td>';
                                    echo '</tr>';
                                }
                            }

                        } catch(PDOException $e) {
                            echo '<tr><td colspan="5" class="text-center text-danger">Erreur lors de la récupération des activités</td></tr>';
                            error_log('Erreur activités newsletter: ' . $e->getMessage());
                        }
                        ?>
                    </tbody>
                </table>
                <div class="text-end">
                    <a href="logs/activity.php" class="btn btn-outline-primary">
                        <i class="fas fa-list me-1"></i>Voir toutes les activités
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Quatrième ligne : Listes de contacts et campagnes récentes -->
    <div class="row mt-4">
        <!-- Listes récentes -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-list me-2"></i>Listes de contacts</h5>
                </div>
                <div class="card-body">
                    <div class="list-group list-group-flush">
                        <?php
                        try {
                            // Récupérer les segments créés (listes dynamiques)
                            $query = "
                                SELECT 
                                    s.name,
                                    s.description,
                                    s.created_at,
                                    'Active' as status,
                                    (SELECT COUNT(*) FROM newsletter_contacts WHERE consent_status IN ('b2b_cold', 'opt_in')) as contact_count
                                FROM newsletter_segments s
                                ORDER BY s.created_at DESC
                                LIMIT 4
                            ";
                            
                            $stmt = $db->query($query);
                            $lists = $stmt->fetchAll(PDO::FETCH_ASSOC);

                            if (empty($lists)) {
                                // Afficher info si pas de segments
                                echo '<div class="text-center py-4">';
                                echo '<i class="fas fa-info-circle text-muted me-2"></i>';
                                echo 'Aucune liste créée. Les contacts importés seront visibles ici.';
                                echo '</div>';
                            } else {
                                foreach($lists as $list) {
                                    $badge_class = $list['status'] == 'Active' ? 'bg-success' : 'bg-warning';
                                    echo '<div class="list-group-item d-flex justify-content-between align-items-center">';
                                    echo '<div>';
                                    echo '<strong>' . htmlspecialchars($list['name']) . '</strong>';
                                    echo '<small class="text-muted d-block">' . number_format($list['contact_count']) . ' contacts</small>';
                                    echo '</div>';
                                    echo '<div>';
                                    echo '<span class="badge ' . $badge_class . '">' . htmlspecialchars($list['status']) . '</span>';
                                    echo '<button class="btn btn-sm btn-outline-primary ms-2">';
                                    echo '<i class="fas fa-edit"></i>';
                                    echo '</button>';
                                    echo '</div>';
                                    echo '</div>';
                                }
                            }

                        } catch(PDOException $e) {
                            echo '<div class="text-center text-danger py-3">Erreur lors du chargement des listes</div>';
                            error_log('Erreur listes newsletter: ' . $e->getMessage());
                        }
                        ?>
                    <div class="text-center mt-3">
                        <a href="lists/index.php" class="btn btn-outline-primary">
                            <i class="fas fa-list me-1"></i>Voir toutes les listes
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Campagnes récentes -->
        <div class="col-md-6">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-paper-plane me-2"></i>Campagnes récentes</h5>
                </div>
                <div class="card-body">
                    <div class="list-group list-group-flush">
                        <?php
                        try {
                            // Récupérer les campagnes récentes
                            $query = "
                                SELECT 
                                    c.name,
                                    c.status,
                                    c.finished_at,
                                    COUNT(d.id) as total_sent,
                                    COUNT(CASE WHEN d.first_open_at IS NOT NULL THEN 1 END) as total_opened,
                                    COUNT(CASE WHEN d.first_click_at IS NOT NULL THEN 1 END) as total_clicked
                                FROM newsletter_campaigns c
                                LEFT JOIN newsletter_deliveries d ON c.id = d.campaign_id
                                WHERE c.status IN ('finished', 'sending')
                                GROUP BY c.id
                                ORDER BY c.created_at DESC
                                LIMIT 3
                            ";
                            
                            $stmt = $db->query($query);
                            $campaigns = $stmt->fetchAll(PDO::FETCH_ASSOC);

                            if (empty($campaigns)) {
                                // Si pas de campagnes
                                echo '<div class="text-center py-4">';
                                echo '<i class="fas fa-info-circle text-muted me-2"></i>';
                                echo 'Aucune campagne envoyée. Créez votre première newsletter !';
                                echo '</div>';
                            } else {
                                foreach($campaigns as $campaign) {
                                    $open_rate = $campaign['total_sent'] > 0 ? round(($campaign['total_opened'] / $campaign['total_sent']) * 100, 1) : 0;
                                    $click_rate = $campaign['total_sent'] > 0 ? round(($campaign['total_clicked'] / $campaign['total_sent']) * 100, 1) : 0;
                                    $status_date = $campaign['finished_at'] ? date('d/m/Y', strtotime($campaign['finished_at'])) : 'En cours';
                                    
                                    echo '<div class="list-group-item d-flex justify-content-between align-items-center">';
                                    echo '<div>';
                                    echo '<strong>' . htmlspecialchars($campaign['name']) . '</strong>';
                                    echo '<small class="text-muted d-block">Envoyée le ' . $status_date . ' - ' . number_format($campaign['total_sent']) . ' destinataires</small>';
                                    echo '<div class="mt-1">';
                                    echo '<span class="badge bg-success me-1">' . $open_rate . '% ouvert</span>';
                                    echo '<span class="badge bg-info me-1">' . $click_rate . '% cliqué</span>';
                                    echo '</div>';
                                    echo '</div>';
                                    echo '<button class="btn btn-sm btn-outline-primary">';
                                    echo '<i class="fas fa-chart-bar"></i>';
                                    echo '</button>';
                                    echo '</div>';
                                }
                            }

                        } catch(PDOException $e) {
                            echo '<div class="text-center text-danger py-3">Erreur lors du chargement des campagnes</div>';
                            error_log('Erreur campagnes newsletter: ' . $e->getMessage());
                        }
                        ?>
                    <div class="text-center mt-3">
                        <a href="campaigns/index.php" class="btn btn-outline-success">
                            <i class="fas fa-paper-plane me-1"></i>Voir toutes les campagnes
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once dirname(__FILE__) . '/../../includes/footer.php'; ?>