<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once dirname(__FILE__) . '/../../../includes/config.php';
require_once dirname(__FILE__) . '/../../../includes/auth.php';

$page_title = 'Gestion des Contacts Newsletter';
requireAdminAuth();

// Paramètres de pagination et filtres
$page = max(1, intval($_GET['page'] ?? 1));
$perPage = 20;
$search = trim($_GET['search'] ?? '');
$filterConsent = $_GET['consent'] ?? '';
$filterIndustry = $_GET['industry'] ?? '';
$sortBy = $_GET['sort'] ?? 'created_at';
$sortOrder = $_GET['order'] ?? 'DESC';

// Gestion de l'édition en ligne (sans modal)
$editContact = null;
if (isset($_GET['edit']) && intval($_GET['edit'])) {
    $editId = intval($_GET['edit']);
    try {
        $stmt = $db->prepare("SELECT * FROM newsletter_contacts WHERE id = ?");
        $stmt->execute([$editId]);
        $editContact = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Erreur récupération contact: " . $e->getMessage());
    }
}

// Actions sur les contacts
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'edit_contact':
                handleEditContact($db, $_POST);
                break;
                
            case 'delete_single':
                handleDeleteSingle($db, $_POST['contact_id']);
                break;
                
            default:
                if (isset($_POST['contact_ids'])) {
                    $contactIds = array_filter(array_map('intval', $_POST['contact_ids']));
                    if (!empty($contactIds)) {
                        handleBulkAction($db, $_POST['action'], $contactIds);
                    }
                }
        }
    }
}

require_once dirname(__FILE__) . '/../../../includes/header.php';

// Récupérer les statistiques
$stats = getContactsStats($db);

// Récupérer les contacts avec filtres
$contacts = getFilteredContacts($db, $page, $perPage, $search, $filterConsent, $filterIndustry, $sortBy, $sortOrder);
$totalContacts = getTotalContacts($db, $search, $filterConsent, $filterIndustry);
$totalPages = ceil($totalContacts / $perPage);

// Récupérer les valeurs pour les filtres
$industries = getUniqueValues($db, 'industry');
?>

<style>
.contacts-header {
    background: linear-gradient(135deg, #2c5aa0 0%, #34495e 100%);
    color: white;
    padding: 2rem 0;
    margin-bottom: 2rem;
}

.contact-row:hover {
    background-color: #f8f9fa;
}

.contact-avatar {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    color: white;
    font-size: 16px;
}

.contact-actions {
    opacity: 0;
    transition: opacity 0.2s;
}

.contact-row:hover .contact-actions {
    opacity: 1;
}

.consent-badge {
    font-size: 0.75rem;
}

.bulk-actions {
    background: #f8f9fa;
    border-radius: 8px;
    padding: 1rem;
    margin-bottom: 1rem;
    display: none;
}

.bulk-actions.show {
    display: block;
}

.edit-row {
    background-color: #fff3cd !important;
    border-left: 4px solid #ffc107;
}

.edit-form {
    background-color: #f8f9fa;
    padding: 1rem;
    border-radius: 8px;
    margin-bottom: 1rem;
}
</style>

<div class="contacts-header">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-md-8">
                <h1><i class="fas fa-users me-3"></i>Gestion des Contacts</h1>
                <p class="mb-0">Gérez votre base de contacts newsletter</p>
            </div>
            <div class="col-md-4 text-end">
                <a href="../import/upload.php" class="btn btn-light btn-lg">
                    <i class="fas fa-plus me-2"></i>Importer des contacts
                </a>
            </div>
        </div>
    </div>
</div>

<div class="container">
    <!-- Statistiques rapides -->
    <div class="row mb-4">
        <div class="col-md-3">
            <div class="card text-center">
                <div class="card-body">
                    <h3 class="text-primary"><?php echo number_format($stats['total']); ?></h3>
                    <small>Contacts total</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card text-center">
                <div class="card-body">
                    <h3 class="text-success"><?php echo number_format($stats['opt_in']); ?></h3>
                    <small>Opt-in</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card text-center">
                <div class="card-body">
                    <h3 class="text-warning"><?php echo number_format($stats['b2b_cold']); ?></h3>
                    <small>B2B Cold</small>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card text-center">
                <div class="card-body">
                    <h3 class="text-danger"><?php echo number_format($stats['opt_out']); ?></h3>
                    <small>Opt-out</small>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulaire d'édition (affiché quand on clique sur modifier) -->
    <?php if ($editContact): ?>
        <div class="edit-form">
            <h5><i class="fas fa-edit me-2"></i>Modification du contact : <?php echo htmlspecialchars($editContact['email']); ?></h5>
            <form method="POST">
                <input type="hidden" name="action" value="edit_contact">
                <input type="hidden" name="contact_id" value="<?php echo $editContact['id']; ?>">
                
                <div class="row">
                    <div class="col-md-3">
                        <div class="mb-3">
                            <label class="form-label">Email</label>
                            <input type="email" class="form-control" name="email" value="<?php echo htmlspecialchars($editContact['email']); ?>" required>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Prénom</label>
                            <input type="text" class="form-control" name="firstname" value="<?php echo htmlspecialchars($editContact['firstname'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Nom</label>
                            <input type="text" class="form-control" name="lastname" value="<?php echo htmlspecialchars($editContact['lastname'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="mb-3">
                            <label class="form-label">Entreprise</label>
                            <input type="text" class="form-control" name="company" value="<?php echo htmlspecialchars($editContact['company'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Consentement</label>
                            <select class="form-select" name="consent_status">
                                <option value="opt_in" <?php echo $editContact['consent_status'] === 'opt_in' ? 'selected' : ''; ?>>Opt-in</option>
                                <option value="b2b_cold" <?php echo $editContact['consent_status'] === 'b2b_cold' ? 'selected' : ''; ?>>B2B Cold</option>
                                <option value="opt_out" <?php echo $editContact['consent_status'] === 'opt_out' ? 'selected' : ''; ?>>Opt-out</option>
                            </select>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Rôle</label>
                            <input type="text" class="form-control" name="role" value="<?php echo htmlspecialchars($editContact['role'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Secteur</label>
                            <input type="text" class="form-control" name="industry" value="<?php echo htmlspecialchars($editContact['industry'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Ville</label>
                            <input type="text" class="form-control" name="city" value="<?php echo htmlspecialchars($editContact['city'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Pays</label>
                            <input type="text" class="form-control" name="country" value="<?php echo htmlspecialchars($editContact['country'] ?? ''); ?>">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Tags</label>
                            <input type="text" class="form-control" name="tags" value="<?php echo htmlspecialchars($editContact['tags'] ?? ''); ?>" placeholder="tag1,tag2">
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="mb-3">
                            <label class="form-label">Source</label>
                            <input type="text" class="form-control" name="source" value="<?php echo htmlspecialchars($editContact['source'] ?? ''); ?>">
                        </div>
                    </div>
                </div>
                
                <div class="d-flex gap-2">
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-save me-1"></i>Enregistrer
                    </button>
                    <a href="contacts.php" class="btn btn-secondary">
                        <i class="fas fa-times me-1"></i>Annuler
                    </a>
                </div>
            </form>
        </div>
    <?php endif; ?>

    <!-- Filtres et recherche -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" class="row align-items-end">
                <div class="col-md-4">
                    <label class="form-label">Rechercher</label>
                    <input type="text" class="form-control" name="search" 
                           value="<?php echo htmlspecialchars($search); ?>" 
                           placeholder="Email, nom, entreprise...">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Consentement</label>
                    <select class="form-select" name="consent">
                        <option value="">Tous</option>
                        <option value="opt_in" <?php echo $filterConsent === 'opt_in' ? 'selected' : ''; ?>>Opt-in</option>
                        <option value="b2b_cold" <?php echo $filterConsent === 'b2b_cold' ? 'selected' : ''; ?>>B2B Cold</option>
                        <option value="opt_out" <?php echo $filterConsent === 'opt_out' ? 'selected' : ''; ?>>Opt-out</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Secteur</label>
                    <select class="form-select" name="industry">
                        <option value="">Tous</option>
                        <?php foreach ($industries as $industry): ?>
                            <option value="<?php echo htmlspecialchars($industry); ?>" 
                                    <?php echo $filterIndustry === $industry ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($industry); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Trier par</label>
                    <select class="form-select" name="sort">
                        <option value="created_at" <?php echo $sortBy === 'created_at' ? 'selected' : ''; ?>>Date création</option>
                        <option value="lastname" <?php echo $sortBy === 'lastname' ? 'selected' : ''; ?>>Nom</option>
                        <option value="company" <?php echo $sortBy === 'company' ? 'selected' : ''; ?>>Entreprise</option>
                        <option value="email" <?php echo $sortBy === 'email' ? 'selected' : ''; ?>>Email</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search me-1"></i>Filtrer
                    </button>
                    <?php if ($search || $filterConsent || $filterIndustry): ?>
                        <a href="contacts.php" class="btn btn-outline-secondary ms-1">
                            <i class="fas fa-times"></i>
                        </a>
                    <?php endif; ?>
                </div>
            </form>
        </div>
    </div>

    <!-- Actions en lot -->
    <div class="bulk-actions" id="bulkActions">
        <form method="POST" id="bulkForm">
            <div class="row align-items-center">
                <div class="col-md-6">
                    <span id="selectedCount">0</span> contact(s) sélectionné(s)
                </div>
                <div class="col-md-6 text-end">
                    <select name="action" class="form-select d-inline-block w-auto me-2" required>
                        <option value="">Choisir une action...</option>
                        <option value="opt_in">Marquer comme Opt-in</option>
                        <option value="b2b_cold">Marquer comme B2B Cold</option>
                        <option value="opt_out">Marquer comme Opt-out</option>
                        <option value="delete">Supprimer</option>
                        <option value="export">Exporter</option>
                    </select>
                    <button type="submit" class="btn btn-primary">Appliquer</button>
                    <button type="button" class="btn btn-outline-secondary" onclick="clearSelection()">Annuler</button>
                </div>
            </div>
        </form>
    </div>

    <!-- Liste des contacts -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">
                <i class="fas fa-list me-2"></i>Contacts 
                (<?php echo number_format($totalContacts); ?> total)
            </h5>
            <div>
                <a href="?<?php echo http_build_query(array_merge($_GET, ['order' => $sortOrder === 'ASC' ? 'DESC' : 'ASC'])); ?>" 
                   class="btn btn-sm btn-outline-primary">
                    <i class="fas fa-sort me-1"></i>
                    <?php echo $sortOrder === 'ASC' ? 'Desc' : 'Asc'; ?>
                </a>
            </div>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th width="40">
                                <input type="checkbox" id="selectAll" onchange="toggleAll(this)">
                            </th>
                            <th>Contact</th>
                            <th>Entreprise</th>
                            <th>Secteur</th>
                            <th>Localisation</th>
                            <th>Consentement</th>
                            <th>Source</th>
                            <th>Créé le</th>
                            <th width="100">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($contacts as $contact): ?>
                            <tr class="contact-row <?php echo ($editContact && $editContact['id'] == $contact['id']) ? 'edit-row' : ''; ?>" data-id="<?php echo $contact['id']; ?>">
                                <td>
                                    <input type="checkbox" class="contact-checkbox" 
                                           name="contact_ids[]" value="<?php echo $contact['id']; ?>"
                                           onchange="updateSelection()">
                                </td>
                                <td>
                                    <div class="d-flex align-items-center">
                                        <div class="contact-avatar me-3" style="background-color: <?php echo getContactColor($contact['firstname'] . $contact['lastname']); ?>">
                                            <?php echo getContactInitials($contact['firstname'], $contact['lastname']); ?>
                                        </div>
                                        <div>
                                            <div class="fw-bold">
                                                <?php echo htmlspecialchars(trim($contact['firstname'] . ' ' . $contact['lastname']) ?: 'Sans nom'); ?>
                                                <?php if ($contact['role']): ?>
                                                    <small class="text-muted">- <?php echo htmlspecialchars($contact['role']); ?></small>
                                                <?php endif; ?>
                                            </div>
                                            <div class="text-muted small">
                                                <a href="mailto:<?php echo htmlspecialchars($contact['email']); ?>" class="text-decoration-none">
                                                    <?php echo htmlspecialchars($contact['email']); ?>
                                                </a>
                                            </div>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <?php if ($contact['company']): ?>
                                        <strong><?php echo htmlspecialchars($contact['company']); ?></strong>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($contact['industry']): ?>
                                        <span class="badge bg-light text-dark">
                                            <?php echo htmlspecialchars($contact['industry']); ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <div class="small">
                                        <?php if ($contact['city']): ?>
                                            <i class="fas fa-map-marker-alt me-1"></i>
                                            <?php echo htmlspecialchars($contact['city']); ?>
                                            <?php if ($contact['country']): ?>
                                                , <?php echo htmlspecialchars($contact['country']); ?>
                                            <?php endif; ?>
                                        <?php else: ?>
                                            <span class="text-muted">-</span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <?php
                                    $consentClass = [
                                        'opt_in' => 'bg-success',
                                        'b2b_cold' => 'bg-warning',
                                        'opt_out' => 'bg-danger'
                                    ];
                                    $consentLabel = [
                                        'opt_in' => 'Opt-in',
                                        'b2b_cold' => 'B2B Cold',
                                        'opt_out' => 'Opt-out'
                                    ];
                                    ?>
                                    <span class="badge consent-badge <?php echo $consentClass[$contact['consent_status']] ?? 'bg-secondary'; ?>">
                                        <?php echo $consentLabel[$contact['consent_status']] ?? $contact['consent_status']; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="small text-muted">
                                        <?php echo htmlspecialchars($contact['source'] ?: 'Inconnu'); ?>
                                    </div>
                                </td>
                                <td>
                                    <div class="small">
                                        <?php echo date('d/m/Y', strtotime($contact['created_at'])); ?>
                                        <div class="text-muted">
                                            <?php echo date('H:i', strtotime($contact['created_at'])); ?>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div class="contact-actions">
                                        <a href="?edit=<?php echo $contact['id']; ?>&<?php echo http_build_query($_GET); ?>" 
                                           class="btn btn-sm btn-outline-primary me-1" 
                                           title="Modifier">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        <form method="POST" style="display: inline;" onsubmit="return confirm('Êtes-vous sûr de vouloir supprimer ce contact ?')">
                                            <input type="hidden" name="action" value="delete_single">
                                            <input type="hidden" name="contact_id" value="<?php echo $contact['id']; ?>">
                                            <button type="submit" class="btn btn-sm btn-outline-danger" title="Supprimer">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
            <div class="card-footer">
                <nav aria-label="Pagination contacts">
                    <ul class="pagination justify-content-center mb-0">
                        <?php for ($i = max(1, $page - 2); $i <= min($totalPages, $page + 2); $i++): ?>
                            <li class="page-item <?php echo $i === $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?<?php echo http_build_query(array_merge($_GET, ['page' => $i])); ?>">
                                    <?php echo $i; ?>
                                </a>
                            </li>
                        <?php endfor; ?>
                    </ul>
                </nav>
                <div class="text-center mt-2 small text-muted">
                    Affichage de <?php echo (($page - 1) * $perPage) + 1; ?> à 
                    <?php echo min($page * $perPage, $totalContacts); ?> sur 
                    <?php echo number_format($totalContacts); ?> contacts
                </div>
            </div>
        <?php endif; ?>
    </div>
</div>

<script>
function toggleAll(checkbox) {
    const checkboxes = document.querySelectorAll('.contact-checkbox');
    checkboxes.forEach(cb => cb.checked = checkbox.checked);
    updateSelection();
}

function updateSelection() {
    const selected = document.querySelectorAll('.contact-checkbox:checked');
    const count = selected.length;
    
    document.getElementById('selectedCount').textContent = count;
    document.getElementById('bulkActions').classList.toggle('show', count > 0);
    
    // Mettre à jour les checkboxes dans le form
    const form = document.getElementById('bulkForm');
    form.querySelectorAll('input[name="contact_ids[]"]').forEach(input => input.remove());
    
    selected.forEach(checkbox => {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'contact_ids[]';
        input.value = checkbox.value;
        form.appendChild(input);
    });
}

function clearSelection() {
    document.querySelectorAll('.contact-checkbox').forEach(cb => cb.checked = false);
    document.getElementById('selectAll').checked = false;
    updateSelection();
}
</script>

<?php
/**
 * Récupérer les statistiques des contacts
 */
function getContactsStats($db) {
    $stats = [
        'total' => 0,
        'opt_in' => 0,
        'b2b_cold' => 0,
        'opt_out' => 0
    ];
    
    try {
        $stmt = $db->query("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN consent_status = 'opt_in' THEN 1 ELSE 0 END) as opt_in,
                SUM(CASE WHEN consent_status = 'b2b_cold' THEN 1 ELSE 0 END) as b2b_cold,
                SUM(CASE WHEN consent_status = 'opt_out' THEN 1 ELSE 0 END) as opt_out
            FROM newsletter_contacts
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $stats = array_merge($stats, $result);
    } catch (PDOException $e) {
        error_log("Erreur stats contacts: " . $e->getMessage());
    }
    
    return $stats;
}

/**
 * Récupérer les contacts filtrés
 */
function getFilteredContacts($db, $page, $perPage, $search, $filterConsent, $filterIndustry, $sortBy, $sortOrder) {
    $offset = ($page - 1) * $perPage;
    $conditions = [];
    $params = [];
    
    if ($search) {
        $conditions[] = "(email LIKE ? OR firstname LIKE ? OR lastname LIKE ? OR company LIKE ?)";
        $searchTerm = '%' . $search . '%';
        $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
    }
    
    if ($filterConsent) {
        $conditions[] = "consent_status = ?";
        $params[] = $filterConsent;
    }
    
    if ($filterIndustry) {
        $conditions[] = "industry = ?";
        $params[] = $filterIndustry;
    }
    
    $whereClause = empty($conditions) ? '' : 'WHERE ' . implode(' AND ', $conditions);
    $orderClause = "ORDER BY " . $sortBy . " " . $sortOrder;
    
    $sql = "
        SELECT * FROM newsletter_contacts 
        {$whereClause} 
        {$orderClause} 
        LIMIT {$perPage} OFFSET {$offset}
    ";
    
    try {
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Erreur récupération contacts: " . $e->getMessage());
        return [];
    }
}

/**
 * Compter le total des contacts filtrés
 */
function getTotalContacts($db, $search, $filterConsent, $filterIndustry) {
    $conditions = [];
    $params = [];
    
    if ($search) {
        $conditions[] = "(email LIKE ? OR firstname LIKE ? OR lastname LIKE ? OR company LIKE ?)";
        $searchTerm = '%' . $search . '%';
        $params = array_merge($params, [$searchTerm, $searchTerm, $searchTerm, $searchTerm]);
    }
    
    if ($filterConsent) {
        $conditions[] = "consent_status = ?";
        $params[] = $filterConsent;
    }
    
    if ($filterIndustry) {
        $conditions[] = "industry = ?";
        $params[] = $filterIndustry;
    }
    
    $whereClause = empty($conditions) ? '' : 'WHERE ' . implode(' AND ', $conditions);
    
    try {
        $stmt = $db->prepare("SELECT COUNT(*) FROM newsletter_contacts {$whereClause}");
        $stmt->execute($params);
        return $stmt->fetchColumn();
    } catch (PDOException $e) {
        error_log("Erreur comptage contacts: " . $e->getMessage());
        return 0;
    }
}

/**
 * Récupérer les valeurs uniques d'un champ
 */
function getUniqueValues($db, $field) {
    try {
        $stmt = $db->prepare("SELECT DISTINCT {$field} FROM newsletter_contacts WHERE {$field} IS NOT NULL AND {$field} != '' ORDER BY {$field}");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    } catch (PDOException $e) {
        error_log("Erreur récupération valeurs uniques: " . $e->getMessage());
        return [];
    }
}

/**
 * Traiter l'édition d'un contact
 */
function handleEditContact($db, $data) {
    try {
        $stmt = $db->prepare("
            UPDATE newsletter_contacts SET 
                email = ?, firstname = ?, lastname = ?, company = ?, role = ?, 
                industry = ?, city = ?, country = ?, tags = ?, consent_status = ?, 
                source = ?, updated_at = NOW()
            WHERE id = ?
        ");
        
        $result = $stmt->execute([
            strtolower($data['email']),
            ucfirst(strtolower($data['firstname'] ?? '')),
            strtoupper($data['lastname'] ?? ''),
            strtoupper($data['company'] ?? ''),
            ucwords(strtolower($data['role'] ?? '')),
            ucwords(strtolower($data['industry'] ?? '')),
            ucfirst(strtolower($data['city'] ?? '')),
            ucfirst(strtolower($data['country'] ?? '')),
            $data['tags'] ?? '',
            $data['consent_status'] ?? 'b2b_cold',
            $data['source'] ?? '',
            intval($data['contact_id'])
        ]);
        
        if ($result) {
            // Redirection pour éviter la resoumission
            header('Location: contacts.php?edited=1');
            exit;
        }
    } catch (PDOException $e) {
        error_log("Erreur édition contact: " . $e->getMessage());
    }
}

/**
 * Traiter la suppression d'un contact
 */
function handleDeleteSingle($db, $contactId) {
    try {
        $stmt = $db->prepare("DELETE FROM newsletter_contacts WHERE id = ?");
        $stmt->execute([intval($contactId)]);
    } catch (PDOException $e) {
        error_log("Erreur suppression contact: " . $e->getMessage());
    }
}

/**
 * Traiter les actions en lot
 */
function handleBulkAction($db, $action, $contactIds) {
    $placeholders = implode(',', array_fill(0, count($contactIds), '?'));
    
    try {
        switch ($action) {
            case 'opt_in':
            case 'b2b_cold':
            case 'opt_out':
                $stmt = $db->prepare("UPDATE newsletter_contacts SET consent_status = ?, updated_at = NOW() WHERE id IN ($placeholders)");
                $stmt->execute(array_merge([$action], $contactIds));
                break;
                
            case 'delete':
                $stmt = $db->prepare("DELETE FROM newsletter_contacts WHERE id IN ($placeholders)");
                $stmt->execute($contactIds);
                break;
                
            case 'export':
                // TODO: Implémenter l'export
                break;
        }
    } catch (PDOException $e) {
        error_log("Erreur action en lot: " . $e->getMessage());
    }
}

/**
 * Générer les initiales d'un contact
 */
function getContactInitials($firstname, $lastname) {
    $initials = '';
    if ($firstname) $initials .= strtoupper(substr($firstname, 0, 1));
    if ($lastname) $initials .= strtoupper(substr($lastname, 0, 1));
    return $initials ?: '?';
}

/**
 * Générer une couleur pour l'avatar d'un contact
 */
function getContactColor($name) {
    $colors = [
        '#3498db', '#e74c3c', '#2ecc71', '#f39c12', '#9b59b6',
        '#1abc9c', '#34495e', '#16a085', '#27ae60', '#2980b9',
        '#8e44ad', '#2c3e50', '#f1c40f', '#e67e22', '#e74c3c'
    ];
    
    $hash = crc32($name);
    return $colors[abs($hash) % count($colors)];
}

require_once dirname(__FILE__) . '/../../../includes/footer.php';
?>