<?php
// =========================================
// FICHIER: /gestionstock/clients/edit.php
// Édition détaillée d'un client
// =========================================

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

require_once '../includes/init.php';
require_once '../includes/header.php';

// Vérifier que l'ID est fourni
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    $_SESSION['error'] = 'Client introuvable';
    header('Location: index.php');
    exit;
}

$client_id = (int)$_GET['id'];

// Récupérer les données du client
$sql = "SELECT * FROM clients WHERE id = ?";
$stmt = $db->prepare($sql);
$stmt->execute([$client_id]);
$client = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$client) {
    $_SESSION['error'] = 'Client introuvable';
    header('Location: index.php');
    exit;
}

// Récupérer les stands du client
$stands_sql = "
    SELECT 
        s.*,
        es.event_id,
        e.nom as event_nom,
        e.lieu as event_lieu,
        e.date_debut as event_date_debut
    FROM stands s
    LEFT JOIN event_stands es ON s.id = es.stand_id
    LEFT JOIN events e ON es.event_id = e.id
    WHERE s.client_id = ?
    ORDER BY s.nom
";
$stands_stmt = $db->prepare($stands_sql);
$stands_stmt->execute([$client_id]);
$stands = $stands_stmt->fetchAll(PDO::FETCH_ASSOC);

// Messages de session
$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Édition Client - <?= htmlspecialchars($client['societe']) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Navigation -->
    <nav class="navbar navbar-expand-lg navbar-dark bg-primary">
        <div class="container-fluid">
            <a class="navbar-brand" href="../"><i class="fas fa-cube"></i> BeMatrix</a>
            <ol class="breadcrumb mb-0">
                <li class="breadcrumb-item"><a href="../" class="text-white text-decoration-none">Accueil</a></li>
                <li class="breadcrumb-item"><a href="index.php" class="text-white text-decoration-none">Clients</a></li>
                <li class="breadcrumb-item active text-light"><?= htmlspecialchars($client['societe']) ?></li>
            </ol>
        </div>
    </nav>

    <div class="container-fluid py-4">
        
        <!-- Messages -->
        <?php if($success): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle"></i> <?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if($error): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle"></i> <?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>

        <!-- Header -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <h1><i class="fas fa-building text-primary"></i> <?= htmlspecialchars($client['societe']) ?></h1>
                        <p class="text-muted mb-0">
                            <i class="fas fa-map-marker-alt"></i> <?= htmlspecialchars($client['pays']) ?>
                            <?php if($client['created_at']): ?>
                            • <i class="fas fa-calendar"></i> Client depuis le <?= date('d/m/Y', strtotime($client['created_at'])) ?>
                            <?php endif; ?>
                        </p>
                    </div>
                    <div>
                        <a href="index.php" class="btn btn-outline-secondary me-2">
                            <i class="fas fa-arrow-left"></i> Retour
                        </a>
                        <button class="btn btn-primary" onclick="showEditModal()">
                            <i class="fas fa-edit"></i> Modifier
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            
            <!-- Colonne de gauche : Informations client -->
            <div class="col-lg-4">
                
                <!-- Informations principales -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5><i class="fas fa-info-circle"></i> Informations Générales</h5>
                    </div>
                    <div class="card-body">
                        <div class="row mb-3">
                            <div class="col-sm-4"><strong>Société :</strong></div>
                            <div class="col-sm-8"><?= htmlspecialchars($client['societe']) ?></div>
                        </div>
                        <div class="row mb-3">
                            <div class="col-sm-4"><strong>Pays :</strong></div>
                            <div class="col-sm-8">
                                <span class="badge bg-primary"><?= htmlspecialchars($client['pays']) ?></span>
                            </div>
                        </div>
                        <?php if($client['secteur_activite']): ?>
                        <div class="row mb-3">
                            <div class="col-sm-4"><strong>Secteur :</strong></div>
                            <div class="col-sm-8"><?= htmlspecialchars($client['secteur_activite']) ?></div>
                        </div>
                        <?php endif; ?>
                        <div class="row mb-3">
                            <div class="col-sm-4"><strong>Statut :</strong></div>
                            <div class="col-sm-8">
                                <?php 
                                $badge_class = match($client['statut'] ?? 'client_actif') {
                                    'client_actif' => 'bg-success',
                                    'prospect' => 'bg-warning text-dark',
                                    'client_inactif' => 'bg-secondary',
                                    default => 'bg-success'
                                };
                                ?>
                                <span class="badge <?= $badge_class ?>"><?= ucfirst(str_replace('_', ' ', $client['statut'] ?? 'Client Actif')) ?></span>
                            </div>
                        </div>
                        <?php if($client['priorite'] && $client['priorite'] != 'normale'): ?>
                        <div class="row mb-3">
                            <div class="col-sm-4"><strong>Priorité :</strong></div>
                            <div class="col-sm-8">
                                <span class="badge <?= $client['priorite'] == 'vip' ? 'bg-danger' : 'bg-warning text-dark' ?>">
                                    <?= ucfirst($client['priorite']) ?>
                                </span>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Contacts -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5><i class="fas fa-users"></i> Contacts</h5>
                    </div>
                    <div class="card-body">
                        <!-- Contact Commercial -->
                        <h6 class="text-success"><i class="fas fa-user-tie"></i> Contact Commercial</h6>
                        <div class="mb-3">
                            <strong><?= htmlspecialchars($client['contact_commercial_prenom'] . ' ' . $client['contact_commercial_nom']) ?></strong><br>
                            <?php if($client['contact_commercial_fonction']): ?>
                            <small class="text-muted"><?= htmlspecialchars($client['contact_commercial_fonction']) ?></small><br>
                            <?php endif; ?>
                            <?php if($client['contact_commercial_mail']): ?>
                            <a href="mailto:<?= $client['contact_commercial_mail'] ?>">
                                <i class="fas fa-envelope text-muted"></i> <?= htmlspecialchars($client['contact_commercial_mail']) ?>
                            </a><br>
                            <?php endif; ?>
                            <?php if($client['contact_commercial_tel']): ?>
                            <a href="tel:<?= $client['contact_commercial_tel'] ?>">
                                <i class="fas fa-phone text-muted"></i> <?= htmlspecialchars($client['contact_commercial_tel']) ?>
                            </a>
                            <?php endif; ?>
                        </div>

                        <!-- Contact Terrain -->
                        <?php if($client['contact_terrain_nom']): ?>
                        <h6 class="text-warning"><i class="fas fa-hard-hat"></i> Contact Terrain</h6>
                        <div class="mb-0">
                            <strong><?= htmlspecialchars($client['contact_terrain_prenom'] . ' ' . $client['contact_terrain_nom']) ?></strong><br>
                            <?php if($client['contact_terrain_fonction']): ?>
                            <small class="text-muted"><?= htmlspecialchars($client['contact_terrain_fonction']) ?></small><br>
                            <?php endif; ?>
                            <?php if($client['contact_terrain_mail']): ?>
                            <a href="mailto:<?= $client['contact_terrain_mail'] ?>">
                                <i class="fas fa-envelope text-muted"></i> <?= htmlspecialchars($client['contact_terrain_mail']) ?>
                            </a><br>
                            <?php endif; ?>
                            <?php if($client['contact_terrain_tel']): ?>
                            <a href="tel:<?= $client['contact_terrain_tel'] ?>">
                                <i class="fas fa-phone text-muted"></i> <?= htmlspecialchars($client['contact_terrain_tel']) ?>
                            </a>
                            <?php endif; ?>
                        </div>
                        <?php else: ?>
                        <div class="alert alert-light py-2">
                            <small><i class="fas fa-info-circle text-muted"></i> Contact terrain identique au contact commercial</small>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Outils collaboratifs -->
                <?php if($client['dropbox_url'] || $client['becad_url']): ?>
                <div class="card mb-4">
                    <div class="card-header">
                        <h5><i class="fas fa-tools"></i> Outils Collaboratifs</h5>
                    </div>
                    <div class="card-body">
                        <?php if($client['dropbox_url']): ?>
                        <a href="<?= htmlspecialchars($client['dropbox_url']) ?>" target="_blank" class="btn btn-outline-info d-block mb-2">
                            <i class="fab fa-dropbox"></i> Accéder au Dropbox
                        </a>
                        <?php endif; ?>
                        <?php if($client['becad_url']): ?>
                        <a href="<?= htmlspecialchars($client['becad_url']) ?>" target="_blank" class="btn btn-outline-warning d-block">
                            <i class="fas fa-cube"></i> Voir en 3D (BECAD)
                        </a>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>

                <!-- Notes internes -->
                <?php if($client['notes_internes']): ?>
                <div class="card">
                    <div class="card-header">
                        <h5><i class="fas fa-sticky-note"></i> Notes Internes</h5>
                    </div>
                    <div class="card-body">
                        <?= nl2br(htmlspecialchars($client['notes_internes'])) ?>
                    </div>
                </div>
                <?php endif; ?>
            </div>

            <!-- Colonne de droite : Stands et activité -->
            <div class="col-lg-8">
                
                <!-- Statistiques rapides -->
                <div class="row mb-4">
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <i class="fas fa-cube fa-2x text-primary mb-2"></i>
                                <h4><?= count($stands) ?></h4>
                                <small>Stand<?= count($stands) > 1 ? 's' : '' ?></small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <i class="fas fa-calendar fa-2x text-info mb-2"></i>
                                <h4><?= count(array_filter($stands, fn($s) => $s['event_id'])) ?></h4>
                                <small>Événement<?= count(array_filter($stands, fn($s) => $s['event_id'])) > 1 ? 's' : '' ?></small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <i class="fas fa-euro-sign fa-2x text-success mb-2"></i>
                                <h4><?= number_format(array_sum(array_column($stands, 'prix_location_ht')), 0, ',', ' ') ?>€</h4>
                                <small>CA Potentiel</small>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="card text-center">
                            <div class="card-body">
                                <i class="fas fa-square fa-2x text-warning mb-2"></i>
                                <h4><?= array_sum(array_column($stands, 'nb_plaques_pvc_supplementaires')) ?></h4>
                                <small>Plaques PVC</small>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Liste des stands -->
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5><i class="fas fa-cube"></i> Stands du Client</h5>
                        <a href="../stands/create.php?client_id=<?= $client_id ?>" class="btn btn-sm btn-success">
                            <i class="fas fa-plus"></i> Nouveau Stand
                        </a>
                    </div>
                    <div class="card-body">
                        <?php if(empty($stands)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-cube fa-3x text-muted mb-3"></i>
                            <h5>Aucun stand configuré</h5>
                            <p class="text-muted">Ce client n'a pas encore de stands configurés.</p>
                            <a href="../stands/create.php?client_id=<?= $client_id ?>" class="btn btn-primary">
                                <i class="fas fa-plus"></i> Créer le premier stand
                            </a>
                        </div>
                        <?php else: ?>
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-light">
                                    <tr>
                                        <th>Stand</th>
                                        <th>Surface</th>
                                        <th>Prix HT</th>
                                        <th>Détails Techniques</th>
                                        <th>Événement</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach($stands as $stand): ?>
                                    <tr>
                                        <td>
                                            <strong><?= htmlspecialchars($stand['nom']) ?></strong>
                                            <?php if($stand['description']): ?>
                                            <br><small class="text-muted"><?= htmlspecialchars(substr($stand['description'], 0, 50)) ?>...</small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?= $stand['surface'] ?>m²</td>
                                        <td>
                                            <?php if($stand['prix_location_ht']): ?>
                                            <strong><?= number_format($stand['prix_location_ht'], 0, ',', ' ') ?>€</strong>
                                            <?php else: ?>
                                            <small class="text-muted">Non défini</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if($stand['cote_poignee_porte'] || $stand['nb_plaques_pvc_supplementaires']): ?>
                                            <small>
                                                <?php if($stand['cote_poignee_porte']): ?>
                                                <i class="fas fa-door-open"></i> Porte <?= $stand['cote_poignee_porte'] ?><br>
                                                <?php endif; ?>
                                                <?php if($stand['nb_plaques_pvc_supplementaires']): ?>
                                                <i class="fas fa-square"></i> +<?= $stand['nb_plaques_pvc_supplementaires'] ?> PVC
                                                <?php endif; ?>
                                            </small>
                                            <?php else: ?>
                                            <small class="text-muted">Standard</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if($stand['event_nom']): ?>
                                            <strong><?= htmlspecialchars($stand['event_nom']) ?></strong><br>
                                            <small class="text-muted">
                                                <i class="fas fa-map-marker-alt"></i> <?= htmlspecialchars($stand['event_lieu']) ?><br>
                                                <i class="fas fa-calendar"></i> <?= date('d/m/Y', strtotime($stand['event_date_debut'])) ?>
                                            </small>
                                            <?php else: ?>
                                            <small class="text-muted">Pas d'événement assigné</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <a href="../stands/view.php?id=<?= $stand['id'] ?>" class="btn btn-sm btn-outline-info" title="Voir">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <a href="../stands/edit.php?id=<?= $stand['id'] ?>" class="btn btn-sm btn-outline-primary" title="Modifier">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal d'édition -->
    <div class="modal fade" id="modalEditClient" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-edit"></i> Modifier <?= htmlspecialchars($client['societe']) ?>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="editClientForm" method="POST" action="actions.php">
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="id" value="<?= $client['id'] ?>">
                        
                        <!-- Onglets -->
                        <ul class="nav nav-tabs" role="tablist">
                            <li class="nav-item">
                                <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#edit-tab-general">
                                    <i class="fas fa-building"></i> Société
                                </button>
                            </li>
                            <li class="nav-item">
                                <button class="nav-link" data-bs-toggle="tab" data-bs-target="#edit-tab-contacts">
                                    <i class="fas fa-users"></i> Contacts
                                </button>
                            </li>
                            <li class="nav-item">
                                <button class="nav-link" data-bs-toggle="tab" data-bs-target="#edit-tab-tools">
                                    <i class="fas fa-tools"></i> Outils
                                </button>
                            </li>
                        </ul>

                        <div class="tab-content mt-3">
                            
                            <!-- Onglet Société -->
                            <div class="tab-pane active" id="edit-tab-general">
                                <div class="row mb-3">
                                    <div class="col-md-8">
                                        <label class="form-label">Société *</label>
                                        <input type="text" class="form-control" name="societe" value="<?= htmlspecialchars($client['societe']) ?>" required>
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Pays</label>
                                        <select class="form-select" name="pays">
                                            <option value="France" <?= $client['pays'] == 'France' ? 'selected' : '' ?>>France</option>
                                            <option value="Belgique" <?= $client['pays'] == 'Belgique' ? 'selected' : '' ?>>Belgique</option>
                                            <option value="Suisse" <?= $client['pays'] == 'Suisse' ? 'selected' : '' ?>>Suisse</option>
                                            <option value="Luxembourg" <?= $client['pays'] == 'Luxembourg' ? 'selected' : '' ?>>Luxembourg</option>
                                            <option value="Allemagne" <?= $client['pays'] == 'Allemagne' ? 'selected' : '' ?>>Allemagne</option>
                                            <option value="Espagne" <?= $client['pays'] == 'Espagne' ? 'selected' : '' ?>>Espagne</option>
                                            <option value="Italie" <?= $client['pays'] == 'Italie' ? 'selected' : '' ?>>Italie</option>
                                        </select>
                                    </div>
                                </div>

                                <div class="row mb-3">
                                    <div class="col-12">
                                        <label class="form-label">Notes internes</label>
                                        <textarea class="form-control" name="notes_internes" rows="4"><?= htmlspecialchars($client['notes_internes']) ?></textarea>
                                    </div>
                                </div>
                            </div>

                            <!-- Onglet Contacts -->
                            <div class="tab-pane" id="edit-tab-contacts">
                                <!-- Contact Commercial -->
                                <h6 class="text-success mb-3">
                                    <i class="fas fa-user-tie"></i> Contact Commercial
                                </h6>
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Nom *</label>
                                        <input type="text" class="form-control" name="contact_commercial_nom" 
                                               value="<?= htmlspecialchars($client['contact_commercial_nom']) ?>" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Prénom *</label>
                                        <input type="text" class="form-control" name="contact_commercial_prenom" 
                                               value="<?= htmlspecialchars($client['contact_commercial_prenom']) ?>" required>
                                    </div>
                                </div>
                                
                                <div class="row mb-4">
                                    <div class="col-md-6">
                                        <label class="form-label">Email *</label>
                                        <input type="email" class="form-control" name="contact_commercial_mail" 
                                               value="<?= htmlspecialchars($client['contact_commercial_mail']) ?>" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Téléphone</label>
                                        <input type="tel" class="form-control" name="contact_commercial_tel" 
                                               value="<?= htmlspecialchars($client['contact_commercial_tel']) ?>">
                                    </div>
                                </div>

                                <!-- Contact Terrain -->
                                <h6 class="text-warning mb-3">
                                    <i class="fas fa-hard-hat"></i> Contact Terrain 
                                </h6>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Nom</label>
                                        <input type="text" class="form-control" name="contact_terrain_nom" 
                                               value="<?= htmlspecialchars($client['contact_terrain_nom']) ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Prénom</label>
                                        <input type="text" class="form-control" name="contact_terrain_prenom" 
                                               value="<?= htmlspecialchars($client['contact_terrain_prenom']) ?>">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Email</label>
                                        <input type="email" class="form-control" name="contact_terrain_mail" 
                                               value="<?= htmlspecialchars($client['contact_terrain_mail']) ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Téléphone</label>
                                        <input type="tel" class="form-control" name="contact_terrain_tel" 
                                               value="<?= htmlspecialchars($client['contact_terrain_tel']) ?>">
                                    </div>
                                </div>
                            </div>

                            <!-- Onglet Outils -->
                            <div class="tab-pane" id="edit-tab-tools">
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">
                                            <i class="fab fa-dropbox text-info"></i> Dossier Dropbox
                                        </label>
                                        <input type="url" class="form-control" name="dropbox_url" 
                                               value="<?= htmlspecialchars($client['dropbox_url']) ?>">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">
                                            <i class="fas fa-cube text-warning"></i> Lien BECAD 3D
                                        </label>
                                        <input type="url" class="form-control" name="becad_url" 
                                               value="<?= htmlspecialchars($client['becad_url']) ?>">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Langue de communication</label>
                                        <select class="form-select" name="langue_communication">
                                            <option value="fr" <?= ($client['langue_communication'] ?? 'fr') == 'fr' ? 'selected' : '' ?>>Français</option>
                                            <option value="en" <?= ($client['langue_communication'] ?? '') == 'en' ? 'selected' : '' ?>>Anglais</option>
                                            <option value="de" <?= ($client['langue_communication'] ?? '') == 'de' ? 'selected' : '' ?>>Allemand</option>
                                            <option value="es" <?= ($client['langue_communication'] ?? '') == 'es' ? 'selected' : '' ?>>Espagnol</option>
                                            <option value="it" <?= ($client['langue_communication'] ?? '') == 'it' ? 'selected' : '' ?>>Italien</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times"></i> Annuler
                    </button>
                    <button type="button" class="btn btn-primary" onclick="saveEditClient()">
                        <i class="fas fa-save"></i> Enregistrer les modifications
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function showEditModal() {
            new bootstrap.Modal(document.getElementById('modalEditClient')).show();
        }

        function saveEditClient() {
            const form = document.getElementById('editClientForm');
            const formData = new FormData(form);
            
            // Validation
            if (!formData.get('societe') || !formData.get('contact_commercial_nom') || 
                !formData.get('contact_commercial_prenom') || !formData.get('contact_commercial_mail')) {
                alert('Veuillez remplir tous les champs obligatoires (*)');
                return;
            }
            
            // Validation email
            const email = formData.get('contact_commercial_mail');
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                alert('Veuillez saisir un email valide');
                return;
            }
            
            form.submit();
        }
    </script>
</body>
</html>