<?php
// =========================================
// FICHIER: /gestionstock/clients/index.php
// Interface gestion des clients - SANS NAVIGATION INTERNE
// =========================================

require_once '../includes/init.php';
require_once '../includes/header.php';

// Récupération des clients avec statistiques
$sql = "
    SELECT 
        c.*,
        COUNT(DISTINCT s.id) as nb_stands,
        COUNT(DISTINCT es.event_id) as nb_evenements,
        SUM(s.prix_location_ht) as ca_potentiel
    FROM clients c
    LEFT JOIN stands s ON c.id = s.client_id  
    LEFT JOIN event_stands es ON s.id = es.stand_id
    GROUP BY c.id 
    ORDER BY c.societe ASC
";
$clients = $db->query($sql)->fetchAll(PDO::FETCH_ASSOC);

// Statistiques globales simplifiées
$stats_sql = "
    SELECT 
        COUNT(c.id) as total_clients,
        COUNT(CASE WHEN c.statut = 'client_actif' THEN 1 END) as clients_actifs,
        COUNT(DISTINCT s.id) as total_stands
    FROM clients c
    LEFT JOIN stands s ON c.id = s.client_id
";
$stats = $db->query($stats_sql)->fetch(PDO::FETCH_ASSOC);

// Messages de session
$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);
?>

<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Clients - BeMatrix</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
    <style>
        .client-card { transition: transform 0.2s; }
        .client-card:hover { transform: translateY(-2px); }
        .status-badge { font-size: 0.75rem; }
        .dropbox-link:hover, .becad-link:hover { color: #0066cc !important; }
    </style>
</head>
<body>

    <div class="container-fluid py-4">
        
        <!-- Messages de session -->
        <?php if($success): ?>
        <div class="alert alert-success alert-dismissible fade show" role="alert">
            <i class="fas fa-check-circle"></i> <?= htmlspecialchars($success) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <?php if($error): ?>
        <div class="alert alert-danger alert-dismissible fade show" role="alert">
            <i class="fas fa-exclamation-triangle"></i> <?= htmlspecialchars($error) ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php endif; ?>
        
        <!-- Header avec statistiques simplifiées -->
        <div class="row mb-4">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <h1><i class="fas fa-building text-primary"></i> Gestion des Clients</h1>
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalClient">
                        <i class="fas fa-plus"></i> Nouveau Client
                    </button>
                </div>
                
                <!-- Statistiques simplifiées -->
                <div class="row">
                    <div class="col-md-6">
                        <div class="card bg-primary text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-building fa-2x me-3"></i>
                                    <div>
                                        <h3 class="mb-0"><?= $stats['clients_actifs'] ?></h3>
                                        <small>Clients Actifs</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="card bg-success text-white">
                            <div class="card-body">
                                <div class="d-flex align-items-center">
                                    <i class="fas fa-cube fa-2x me-3"></i>
                                    <div>
                                        <h3 class="mb-0"><?= $stats['total_stands'] ?></h3>
                                        <small>Stands Configurés</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filtres simplifiés -->
        <div class="row mb-4">
            <div class="col-md-4">
                <div class="input-group">
                    <span class="input-group-text"><i class="fas fa-search"></i></span>
                    <input type="text" class="form-control" id="searchClient" placeholder="Rechercher société, contact...">
                </div>
            </div>
            <div class="col-md-2">
                <select class="form-select" id="filterStatut">
                    <option value="">Tous statuts</option>
                    <option value="client_actif">Client Actif</option>
                    <option value="prospect">Prospect</option>
                    <option value="client_inactif">Client Inactif</option>
                </select>
            </div>
            <div class="col-md-2">
                <select class="form-select" id="filterPays">
                    <option value="">Tous pays</option>
                    <?php
                    $pays_query = $db->query("SELECT DISTINCT pays FROM clients ORDER BY pays");
                    while($pays = $pays_query->fetch()) {
                        echo "<option value='{$pays['pays']}'>{$pays['pays']}</option>";
                    }
                    ?>
                </select>
            </div>
            <div class="col-md-2">
                <button class="btn btn-outline-secondary" onclick="resetFilters()">
                    <i class="fas fa-times"></i> Reset
                </button>
            </div>
        </div>

        <!-- Liste des clients en cartes simplifiées -->
        <div class="row" id="clientsContainer">
            <?php foreach($clients as $client): ?>
            <div class="col-lg-6 col-xl-4 mb-4 client-item" 
                 data-statut="<?= $client['statut'] ?>" 
                 data-pays="<?= $client['pays'] ?>" 
                 data-priorite="<?= $client['priorite'] ?? 'normale' ?>">
                <div class="card client-card h-100">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0"><strong><?= htmlspecialchars($client['societe']) ?></strong></h6>
                        <div>
                            <?php if(($client['priorite'] ?? '') == 'vip'): ?>
                            <span class="badge bg-danger me-1">VIP</span>
                            <?php elseif(($client['priorite'] ?? '') == 'haute'): ?>
                            <span class="badge bg-warning text-dark me-1">Haute</span>
                            <?php endif; ?>
                            
                            <?php 
                            $badge_class = match($client['statut'] ?? 'client_actif') {
                                'client_actif' => 'bg-success',
                                'prospect' => 'bg-warning text-dark',
                                'client_inactif' => 'bg-secondary',
                                default => 'bg-success'
                            };
                            ?>
                            <span class="badge <?= $badge_class ?>"><?= ucfirst(str_replace('_', ' ', $client['statut'] ?? 'Client Actif')) ?></span>
                        </div>
                    </div>
                    <div class="card-body">
                        <div class="row mb-2">
                            <div class="col-6">
                                <small class="text-muted">Contact Commercial</small><br>
                                <strong><?= htmlspecialchars($client['contact_commercial_prenom'] . ' ' . $client['contact_commercial_nom']) ?></strong><br>
                                <?php if($client['contact_commercial_mail']): ?>
                                <a href="mailto:<?= $client['contact_commercial_mail'] ?>" class="text-decoration-none">
                                    <i class="fas fa-envelope text-muted"></i> <?= htmlspecialchars($client['contact_commercial_mail']) ?>
                                </a><br>
                                <?php endif; ?>
                                <?php if($client['contact_commercial_tel']): ?>
                                <a href="tel:<?= str_replace(['.', ' '], '', $client['contact_commercial_tel']) ?>" class="text-decoration-none">
                                    <i class="fas fa-phone text-muted"></i> <?= htmlspecialchars(str_replace('.', ' ', $client['contact_commercial_tel'])) ?>
                                </a>
                                <?php endif; ?>
                            </div>
                            <div class="col-6">
                                <small class="text-muted">Pays & Contact Terrain</small><br>
                                <span class="badge bg-primary"><?= htmlspecialchars($client['pays']) ?></span><br>
                                <?php if($client['contact_terrain_nom']): ?>
                                <small><strong>Terrain :</strong> <?= htmlspecialchars($client['contact_terrain_prenom'] . ' ' . $client['contact_terrain_nom']) ?></small>
                                <?php else: ?>
                                <small class="text-muted">Même personne</small>
                                <?php endif; ?>
                            </div>
                        </div>

                        <!-- Liens outils -->
                        <?php if($client['dropbox_url'] || $client['becad_url']): ?>
                        <div class="d-flex gap-2 mb-3">
                            <?php if($client['dropbox_url']): ?>
                            <a href="<?= htmlspecialchars($client['dropbox_url']) ?>" target="_blank" 
                               class="btn btn-sm btn-outline-info dropbox-link" title="Dossier Dropbox">
                                <i class="fab fa-dropbox"></i>
                            </a>
                            <?php endif; ?>
                            <?php if($client['becad_url']): ?>
                            <a href="<?= htmlspecialchars($client['becad_url']) ?>" target="_blank" 
                               class="btn btn-sm btn-outline-warning becad-link" title="Vue BECAD 3D">
                                <i class="fas fa-cube"></i>
                            </a>
                            <?php endif; ?>
                        </div>
                        <?php endif; ?>

                        <!-- Notes internes (aperçu) -->
                        <?php if($client['notes_internes']): ?>
                        <div class="alert alert-light py-2">
                            <small><i class="fas fa-sticky-note text-muted"></i> 
                            <?= htmlspecialchars(substr($client['notes_internes'], 0, 60)) ?>
                            <?= strlen($client['notes_internes']) > 60 ? '...' : '' ?>
                            </small>
                        </div>
                        <?php endif; ?>
                    </div>
                    <div class="card-footer">
                        <div class="btn-group w-100">
                            <button class="btn btn-sm btn-outline-primary" onclick="editClient(<?= $client['id'] ?>)">
                                <i class="fas fa-edit"></i>
                            </button>
                            <button class="btn btn-sm btn-outline-info" onclick="viewStands(<?= $client['id'] ?>)">
                                <i class="fas fa-eye"></i> Stands
                            </button>
                            <button class="btn btn-sm btn-outline-danger" onclick="deleteClient(<?= $client['id'] ?>, '<?= htmlspecialchars($client['societe'], ENT_QUOTES) ?>')">
                                <i class="fas fa-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
            
            <!-- Message si aucun client -->
            <?php if(empty($clients)): ?>
            <div class="col-12">
                <div class="card">
                    <div class="card-body text-center py-5">
                        <i class="fas fa-building fa-3x text-muted mb-3"></i>
                        <h5>Aucun client enregistré</h5>
                        <p class="text-muted">Commencez par créer votre premier client en cliquant sur "Nouveau Client"</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#modalClient">
                            <i class="fas fa-plus"></i> Créer le premier client
                        </button>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal Nouveau Client SIMPLIFIÉ -->
    <div class="modal fade" id="modalClient" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">
                        <i class="fas fa-building"></i> <span id="modalTitle">Nouveau Client</span>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="clientForm" method="POST" action="actions.php">
                        <input type="hidden" name="action" value="create">
                        <input type="hidden" id="clientId" name="id">
                        
                        <!-- Onglets simplifiés -->
                        <ul class="nav nav-tabs" role="tablist">
                            <li class="nav-item">
                                <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tab-general">
                                    <i class="fas fa-building"></i> Société
                                </button>
                            </li>
                            <li class="nav-item">
                                <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-contacts">
                                    <i class="fas fa-users"></i> Contacts
                                </button>
                            </li>
                            <li class="nav-item">
                                <button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-tools">
                                    <i class="fas fa-tools"></i> Outils
                                </button>
                            </li>
                        </ul>

                        <div class="tab-content mt-3">
                            
                            <!-- Onglet Société -->
                            <div class="tab-pane active" id="tab-general">
                                <div class="row mb-3">
                                    <div class="col-md-8">
                                        <label class="form-label">Société *</label>
                                        <input type="text" class="form-control" name="societe" required 
                                               placeholder="Nom de la société">
                                    </div>
                                    <div class="col-md-4">
                                        <label class="form-label">Pays</label>
                                        <select class="form-select" name="pays">
                                            <option value="France">France</option>
                                            <option value="Belgique">Belgique</option>
                                            <option value="Suisse">Suisse</option>
                                            <option value="Luxembourg">Luxembourg</option>
                                            <option value="Allemagne">Allemagne</option>
                                            <option value="Espagne">Espagne</option>
                                            <option value="Italie">Italie</option>
                                        </select>
                                    </div>
                                </div>

                                <div class="row mb-3">
                                    <div class="col-12">
                                        <label class="form-label">Notes internes</label>
                                        <textarea class="form-control" name="notes_internes" rows="3" 
                                                  placeholder="Notes pour l'équipe iDkom (contraintes, préférences, historique...)"></textarea>
                                    </div>
                                </div>
                            </div>

                            <!-- Onglet Contacts -->
                            <div class="tab-pane" id="tab-contacts">
                                <!-- Contact Commercial -->
                                <h6 class="text-success mb-3">
                                    <i class="fas fa-user-tie"></i> Contact Commercial
                                    <small class="text-muted">(celui qui gère avec nous)</small>
                                </h6>
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Nom *</label>
                                        <input type="text" class="form-control" name="contact_commercial_nom" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Prénom *</label>
                                        <input type="text" class="form-control" name="contact_commercial_prenom" required>
                                    </div>
                                </div>
                                
                                <div class="row mb-4">
                                    <div class="col-md-6">
                                        <label class="form-label">Email *</label>
                                        <input type="email" class="form-control" name="contact_commercial_mail" required>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Téléphone</label>
                                        <input type="tel" class="form-control" name="contact_commercial_tel" 
                                               placeholder="Ex: 03 84 12 34 56">
                                    </div>
                                </div>

                                <!-- Contact Terrain -->
                                <h6 class="text-warning mb-3">
                                    <i class="fas fa-hard-hat"></i> Contact Terrain 
                                    <small class="text-muted">(celui sur place lors de l'événement)</small>
                                </h6>
                                <div class="alert alert-info py-2 mb-3">
                                    <small><i class="fas fa-info-circle"></i> Optionnel - si différent du contact commercial</small>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Nom</label>
                                        <input type="text" class="form-control" name="contact_terrain_nom">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Prénom</label>
                                        <input type="text" class="form-control" name="contact_terrain_prenom">
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Email</label>
                                        <input type="email" class="form-control" name="contact_terrain_mail">
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">Téléphone</label>
                                        <input type="tel" class="form-control" name="contact_terrain_tel">
                                    </div>
                                </div>
                            </div>

                            <!-- Onglet Outils -->
                            <div class="tab-pane" id="tab-tools">
                                <div class="alert alert-light mb-3">
                                    <i class="fas fa-info-circle text-primary"></i>
                                    <strong>Liens collaboratifs par client</strong><br>
                                    <small>Ces liens seront accessibles depuis tous les stands de ce client</small>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">
                                            <i class="fab fa-dropbox text-info"></i> Dossier Dropbox
                                        </label>
                                        <input type="url" class="form-control" name="dropbox_url" 
                                               placeholder="https://dropbox.com/sh/...">
                                        <small class="text-muted">Dossier partagé pour maquettes, plans, photos</small>
                                    </div>
                                    <div class="col-md-6">
                                        <label class="form-label">
                                            <i class="fas fa-cube text-warning"></i> Lien BECAD 3D
                                        </label>
                                        <input type="url" class="form-control" name="becad_url" 
                                               placeholder="https://becad.com/view/...">
                                        <small class="text-muted">Visualisation 3D des stands du client</small>
                                    </div>
                                </div>
                                
                                <div class="row mb-3">
                                    <div class="col-md-6">
                                        <label class="form-label">Langue de communication</label>
                                        <select class="form-select" name="langue_communication">
                                            <option value="fr">Français</option>
                                            <option value="en">Anglais</option>
                                            <option value="de">Allemand</option>
                                            <option value="es">Espagnol</option>
                                            <option value="it">Italien</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times"></i> Annuler
                    </button>
                    <button type="button" class="btn btn-primary" onclick="saveClient()">
                        <i class="fas fa-save"></i> Enregistrer
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Gestion des filtres
        function applyFilters() {
            const searchTerm = document.getElementById('searchClient').value.toLowerCase();
            const filterStatut = document.getElementById('filterStatut').value;
            const filterPays = document.getElementById('filterPays').value;
            
            document.querySelectorAll('.client-item').forEach(item => {
                const text = item.textContent.toLowerCase();
                const statut = item.dataset.statut;
                const pays = item.dataset.pays;
                
                const matchSearch = !searchTerm || text.includes(searchTerm);
                const matchStatut = !filterStatut || statut === filterStatut;
                const matchPays = !filterPays || pays === filterPays;
                
                item.style.display = (matchSearch && matchStatut && matchPays) ? 'block' : 'none';
            });
        }
        
        // Événements de filtrage
        document.getElementById('searchClient').addEventListener('input', applyFilters);
        document.getElementById('filterStatut').addEventListener('change', applyFilters);
        document.getElementById('filterPays').addEventListener('change', applyFilters);
        
        function resetFilters() {
            document.getElementById('searchClient').value = '';
            document.getElementById('filterStatut').value = '';
            document.getElementById('filterPays').value = '';
            applyFilters();
        }

        // Gestion des actions clients
        function editClient(id) {
            window.location.href = `edit.php?id=${id}`;
        }

        function viewStands(clientId) {
            window.location.href = `../stands/?client_id=${clientId}`;
        }

        function deleteClient(id, societe) {
            if (confirm(`Êtes-vous sûr de vouloir supprimer le client "${societe}" ?\n\nLes stands liés seront dissociés mais conservés.`)) {
                window.location.href = `actions.php?action=delete&id=${id}`;
            }
        }

        function saveClient() {
            const form = document.getElementById('clientForm');
            const formData = new FormData(form);
            
            // Validation des champs obligatoires
            if (!formData.get('societe') || !formData.get('contact_commercial_nom') || 
                !formData.get('contact_commercial_prenom') || !formData.get('contact_commercial_mail')) {
                alert('Veuillez remplir tous les champs obligatoires (*)');
                return;
            }
            
            // Validation email
            const email = formData.get('contact_commercial_mail');
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(email)) {
                alert('Veuillez saisir un email valide');
                return;
            }
            
            // Soumission du formulaire
            form.submit();
        }
    </script>
</body>
</html>