﻿<?php
require_once '../includes/init.php';
requireLogin();

// Récupération des types de stands
try {
    $stmt = $db->query("
        SELECT id, nom, surface, prix_location_base_ht, nb_angles_standard 
        FROM stand_types 
        WHERE is_active = 1 
        ORDER BY surface
    ");
    $types_stands = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    error_log("Erreur lors de la récupération des types de stands : " . $e->getMessage());
    $types_stands = [];
    addMessage("Erreur lors du chargement des types de stands.", "error");
}

// Récupération des événements
try {
    $stmt = $db->query("
        SELECT id, nom, lieu, date_debut, date_fin, client_nom 
        FROM events 
        WHERE date_fin >= CURRENT_DATE 
        AND statut != 'annule'
        ORDER BY date_debut
    ");
    $evenements = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    error_log("Erreur lors de la récupération des événements : " . $e->getMessage());
    $evenements = [];
    addMessage("Erreur lors du chargement des événements.", "error");
}

// Récupération des clients
try {
    $stmt = $db->query("
        SELECT id, societe, contact_commercial_nom, contact_commercial_prenom 
        FROM clients 
        WHERE statut IN ('prospect', 'client')
        ORDER BY societe
    ");
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    error_log("Erreur lors de la récupération des clients : " . $e->getMessage());
    $clients = [];
    addMessage("Erreur lors du chargement des clients.", "error");
}

// Récupération des options de stand
try {
    $stmt = $db->query("
        SELECT id, nom, description, prix_location_ht 
        FROM stand_options 
        WHERE is_active = 1 
        ORDER BY nom
    ");
    $stand_options = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    error_log("Erreur lors de la récupération des options : " . $e->getMessage());
    $stand_options = [];
}

$page_title = "Nouveau stand";
require_once '../includes/header.php';
?>

<div class="container my-4 fade-in">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="mb-0">
            <i class="fas fa-plus-circle me-2 text-primary"></i><?= $page_title ?>
        </h1>
        <a href="../stand/liste.php" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-2"></i>Retour à la liste
        </a>
    </div>

    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger">
            <?= $error_message ?>
        </div>
    <?php endif; ?>

    <form method="POST" action="../stand/save.php" enctype="multipart/form-data" id="standForm" class="needs-validation" novalidate>
        <!-- Sélection du type de création -->
        <div class="card shadow-sm mb-4">
            <div class="card-body">
                <div class="form-check form-check-inline">
                    <input class="form-check-input" type="radio" name="creation_type" id="type_preconfigure" value="preconfigure" checked>
                    <label class="form-check-label" for="type_preconfigure">Stand pré-configuré</label>
                </div>
                <div class="form-check form-check-inline">
                    <input class="form-check-input" type="radio" name="creation_type" id="type_custom" value="custom">
                    <label class="form-check-label" for="type_custom">Stand personnalisé</label>
                </div>
            </div>
        </div>

        <div class="row g-4">
            <!-- Informations de base -->
            <div class="col-md-6">
                <div class="card shadow-sm h-100">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-info-circle me-2 text-primary"></i>
                            Informations générales
                        </h5>
                    </div>
                    <div class="card-body">
                        <!-- Sélection de l'événement -->
                        <div class="mb-3">
                            <label for="evenement" class="form-label">Événement associé</label>
                            <select class="form-select" id="evenement" name="evenement_id">
                                <option value="">Aucun événement</option>
                                <?php foreach ($evenements as $event): ?>
                                    <option value="<?= $event['id'] ?>">
                                        <?= htmlspecialchars($event['nom']) ?> 
                                        <?= !empty($event['client_nom']) ? ' - ' . htmlspecialchars($event['client_nom']) : '' ?>
                                        (<?= date('d/m/Y', strtotime($event['date_debut'])) ?> - 
                                         <?= date('d/m/Y', strtotime($event['date_fin'])) ?>)
                                        <?= $event['lieu'] ? ' - ' . htmlspecialchars($event['lieu']) : '' ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <!-- Sélection du client -->
                        <div class="mb-3">
                            <label for="client_id" class="form-label">Client / Exposant *</label>
                            <div class="input-group">
                                <select class="form-select" id="client_id" name="client_id" required>
                                    <option value="">Sélectionner un client</option>
                                    <?php foreach ($clients as $client): ?>
                                        <option value="<?= $client['id'] ?>">
                                            <?= htmlspecialchars($client['societe']) ?>
                                            <?php if (!empty($client['contact_commercial_nom'])): ?>
                                                - <?= htmlspecialchars($client['contact_commercial_prenom']) ?> <?= htmlspecialchars($client['contact_commercial_nom']) ?>
                                            <?php endif; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <button type="button" class="btn btn-outline-secondary" data-bs-toggle="modal" data-bs-target="#newClientModal" title="Créer un nouveau client">
                                    <i class="fas fa-plus"></i>
                                </button>
                            </div>
                            <div class="invalid-feedback">
                                Le client est requis
                            </div>
                        </div>

                        <!-- Nom du stand -->
                        <div class="mb-3">
                            <label for="nom" class="form-label">Nom du stand *</label>
                            <input type="text" class="form-control" id="nom" name="nom" required
                                   placeholder="Ex: Stand principal, A1, Zone VIP...">
                            <div class="invalid-feedback">
                                Le nom du stand est requis
                            </div>
                            <small class="text-muted">Le nom du client sera automatiquement associé</small>
                        </div>

                        <div class="mb-3 preconfigure-only">
                            <label for="type_stand" class="form-label">Type de stand *</label>
                            <select class="form-select" id="type_stand" name="type_id">
                                <option value="">Sélectionner un type de stand</option>
                                <?php foreach ($types_stands as $type): ?>
                                    <option value="<?= $type['id'] ?>" 
                                            data-surface="<?= $type['surface'] ?>"
                                            data-prix="<?= $type['prix_location_base_ht'] ?>"
                                            data-angles="<?= $type['nb_angles_standard'] ?>">
                                        <?= htmlspecialchars($type['nom']) ?> (<?= $type['surface'] ?>m²)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <div class="invalid-feedback">
                                Le type de stand est requis pour un stand pré-configuré
                            </div>
                        </div>

                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="surface" class="form-label">Surface (m²) *</label>
                                <input type="number" class="form-control" id="surface" name="surface" 
                                       required min="1">
                                <div class="invalid-feedback">
                                    La surface est requise et doit être supérieure à 0
                                </div>
                            </div>
                            <div class="col-md-6">
                                <label for="nb_angles" class="form-label">Nombre d'angles</label>
                                <input type="number" class="form-control" id="nb_angles" 
                                       name="nb_angles" min="0" value="0">
                                <div class="invalid-feedback">
                                    Le nombre d'angles doit être positif ou nul
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="prix_location_ht" class="form-label">Prix location HT *</label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="prix_location_ht" 
                                       name="prix_location_ht" required step="0.01" min="0">
                                <span class="input-group-text">€</span>
                                <div class="invalid-feedback">
                                    Le prix de location est requis et doit être positif
                                </div>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3"></textarea>
                        </div>

                        <div class="mb-3">
                            <label for="image" class="form-label">Image du stand</label>
                            <input type="file" class="form-control" id="image" name="image" 
                                   accept="image/*">
                            <small class="text-muted">Format recommandé : JPG, PNG. Max 5MB</small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Spécifications techniques -->
            <div class="col-md-6">
                <div class="card shadow-sm h-100">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-cogs me-2 text-primary"></i>
                            Spécifications techniques
                        </h5>
                    </div>
                    <div class="card-body">
                        <!-- Code porte -->
                        <div class="mb-3">
                            <label for="code_porte" class="form-label">
                                <i class="fas fa-lock me-2 text-secondary"></i>
                                Code porte (3 chiffres)
                            </label>
                            <input type="text" class="form-control" id="code_porte" name="code_porte" 
                                   maxlength="3" pattern="[0-9]{3}" placeholder="123">
                            <div class="invalid-feedback">
                                Le code doit contenir exactement 3 chiffres
                            </div>
                            <small class="text-muted">Code à 3 chiffres pour ouvrir la serrure</small>
                        </div>

                        <!-- Côté poignée -->
                        <div class="mb-3">
                            <label class="form-label">
                                <i class="fas fa-door-open me-2 text-secondary"></i>
                                Côté de la poignée
                            </label>
                            <div class="btn-group w-100" role="group">
                                <input type="radio" class="btn-check" name="cote_poignee_porte" id="poignee_droite" value="droite">
                                <label class="btn btn-outline-primary" for="poignee_droite">
                                    <i class="fas fa-arrow-right me-2"></i>Droite
                                </label>
                                <input type="radio" class="btn-check" name="cote_poignee_porte" id="poignee_gauche" value="gauche">
                                <label class="btn btn-outline-primary" for="poignee_gauche">
                                    <i class="fas fa-arrow-left me-2"></i>Gauche
                                </label>
                            </div>
                            <small class="text-muted">Position de la poignée vue de l'extérieur</small>
                        </div>

                        <!-- Plaques PVC supplémentaires -->
                        <div class="mb-3">
                            <label for="nb_plaques_pvc_supplementaires" class="form-label">
                                <i class="fas fa-square me-2 text-secondary"></i>
                                Plaques PVC supplémentaires
                            </label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="nb_plaques_pvc_supplementaires" 
                                       name="nb_plaques_pvc_supplementaires" min="0" value="0">
                                <span class="input-group-text">plaques</span>
                            </div>
                            <small class="text-muted">Plaques PVC blanches pour parois communes</small>
                        </div>

                        <!-- Remarques techniques -->
                        <div class="mb-3">
                            <label for="remarques" class="form-label">
                                <i class="fas fa-sticky-note me-2 text-secondary"></i>
                                Remarques techniques
                            </label>
                            <textarea class="form-control" id="remarques" name="remarques" rows="4" 
                                      placeholder="Contraintes de montage, orientations particulières, matériel spécifique..."></textarea>
                            <small class="text-muted">Notes pour l'équipe de montage</small>
                        </div>

                        <!-- Aperçu récapitulatif -->
                        <div class="alert alert-light border">
                            <h6 class="mb-2">
                                <i class="fas fa-eye me-2 text-info"></i>Aperçu du stand
                            </h6>
                            <div id="stand-preview">
                                <small class="text-muted">Les informations apparaîtront ici au fur et à mesure de la saisie</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Options disponibles -->
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-puzzle-piece me-2 text-primary"></i>
                            Options disponibles
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="alert alert-info mb-3">
                            <i class="fas fa-info-circle me-2"></i>
                            Les options pourront être modifiées lors de l'attribution du stand à un événement.
                        </div>
                        <div class="row">
                            <?php foreach ($stand_options as $option): ?>
                                <div class="col-md-6 mb-3">
                                    <div class="card">
                                        <div class="card-body p-3">
                                            <div class="form-check d-flex justify-content-between align-items-center">
                                                <div>
                                                    <input class="form-check-input" type="checkbox" 
                                                           name="options[]" value="<?= $option['id'] ?>" 
                                                           id="option_<?= $option['id'] ?>">
                                                    <label class="form-check-label" for="option_<?= $option['id'] ?>">
                                                        <h6 class="mb-1"><?= htmlspecialchars($option['nom']) ?></h6>
                                                        <?php if ($option['description']): ?>
                                                            <small class="text-muted"><?= htmlspecialchars($option['description']) ?></small>
                                                        <?php endif; ?>
                                                    </label>
                                                </div>
                                                <div class="text-end">
                                                    <?php if ($option['prix_location_ht']): ?>
                                                        <div class="badge bg-secondary">
                                                            <?= number_format($option['prix_location_ht'], 2) ?> € HT
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Boutons de soumission -->
            <div class="col-12">
                <hr>
                <div class="d-flex gap-2 justify-content-end">
                    <button type="submit" class="btn btn-primary btn-lg">
                        <i class="fas fa-save me-2"></i>Enregistrer le stand
                    </button>
                    <a href="../stand/liste.php" class="btn btn-secondary btn-lg">
                        <i class="fas fa-times me-2"></i>Annuler
                    </a>
                </div>
            </div>
        </div>
    </form>
</div>

<!-- Modal pour créer un nouveau client -->
<div class="modal fade" id="newClientModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Nouveau Client</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="quickClientForm">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label for="new_client_societe" class="form-label">Société *</label>
                            <input type="text" class="form-control" id="new_client_societe" required>
                        </div>
                        <div class="col-md-6">
                            <label for="new_client_secteur" class="form-label">Secteur d'activité</label>
                            <input type="text" class="form-control" id="new_client_secteur">
                        </div>
                        <div class="col-md-4">
                            <label for="new_client_nom" class="form-label">Nom contact</label>
                            <input type="text" class="form-control" id="new_client_nom">
                        </div>
                        <div class="col-md-4">
                            <label for="new_client_prenom" class="form-label">Prénom contact</label>
                            <input type="text" class="form-control" id="new_client_prenom">
                        </div>
                        <div class="col-md-4">
                            <label for="new_client_email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="new_client_email">
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Annuler</button>
                <button type="button" class="btn btn-primary" id="saveQuickClient">
                    <i class="fas fa-save me-2"></i>Créer le client
                </button>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('standForm');
    const typeStandSelect = document.getElementById('type_stand');
    const surfaceInput = document.getElementById('surface');
    const prixInput = document.getElementById('prix_location_ht');
    const anglesInput = document.getElementById('nb_angles');
    const creationTypeInputs = document.querySelectorAll('input[name="creation_type"]');
    const preConfigureElements = document.querySelectorAll('.preconfigure-only');
    const codePorteInput = document.getElementById('code_porte');
    const previewDiv = document.getElementById('stand-preview');

    // Fonction pour générer un aperçu du stand
    function updateStandPreview() {
        const client = document.getElementById('client_id').selectedOptions[0]?.textContent || 'Non défini';
        const nom = document.getElementById('nom').value || 'Non défini';
        const surface = document.getElementById('surface').value || '-';
        const prix = document.getElementById('prix_location_ht').value || '-';
        const codePorte = document.getElementById('code_porte').value || 'Non défini';
        const cotePoignee = document.querySelector('input[name="cote_poignee_porte"]:checked')?.value || 'Non défini';
        const plaquesPvc = document.getElementById('nb_plaques_pvc_supplementaires').value || '0';
        
        previewDiv.innerHTML = `
            <div class="row g-2">
                <div class="col-6"><strong>Client:</strong> ${client}</div>
                <div class="col-6"><strong>Stand:</strong> ${nom}</div>
                <div class="col-6"><strong>Surface:</strong> ${surface} m²</div>
                <div class="col-6"><strong>Prix:</strong> ${prix} € HT</div>
                <div class="col-6"><strong>Code porte:</strong> ${codePorte}</div>
                <div class="col-6"><strong>Poignée:</strong> ${cotePoignee}</div>
                <div class="col-12"><strong>Plaques PVC:</strong> ${plaquesPvc} supplémentaire(s)</div>
            </div>
        `;
    }

    // Validation du code porte (3 chiffres uniquement)
    codePorteInput.addEventListener('input', function(e) {
        e.target.value = e.target.value.replace(/[^0-9]/g, '').substring(0, 3);
        updateStandPreview();
    });

    // Fonction pour gérer l'affichage des champs selon le type de création
    function toggleCreationType(isCustom) {
        preConfigureElements.forEach(element => {
            element.style.display = isCustom ? 'none' : 'block';
        });

        // Réinitialiser et gérer les champs requis
        if (isCustom) {
            typeStandSelect.removeAttribute('required');
            surfaceInput.removeAttribute('readonly');
            prixInput.removeAttribute('readonly');
            anglesInput.removeAttribute('readonly');
            
            // Réinitialiser les valeurs en mode personnalisé
            surfaceInput.value = '';
            prixInput.value = '';
            anglesInput.value = '0';
            typeStandSelect.value = '';
        } else {
            typeStandSelect.setAttribute('required', '');
            surfaceInput.setAttribute('readonly', '');
            prixInput.setAttribute('readonly', '');
            anglesInput.setAttribute('readonly', '');
        }
        updateStandPreview();
    }

    // Écouter le changement de type de création
    creationTypeInputs.forEach(input => {
        input.addEventListener('change', (e) => {
            const isCustom = e.target.value === 'custom';
            toggleCreationType(isCustom);
        });
    });

    // Gestion du changement de type de stand
    typeStandSelect.addEventListener('change', function(e) {
        const selectedOption = e.target.options[e.target.selectedIndex];
        
        if (selectedOption.value) {
            surfaceInput.value = selectedOption.dataset.surface;
            prixInput.value = selectedOption.dataset.prix;
            anglesInput.value = selectedOption.dataset.angles;
        } else {
            surfaceInput.value = '';
            prixInput.value = '';
            anglesInput.value = '';
        }
        updateStandPreview();
    });

    // Mettre à jour l'aperçu lors de changements
    ['client_id', 'nom', 'surface', 'prix_location_ht', 'nb_plaques_pvc_supplementaires'].forEach(id => {
        document.getElementById(id).addEventListener('change', updateStandPreview);
        document.getElementById(id).addEventListener('input', updateStandPreview);
    });

    // Écouter les changements de côté de poignée
    document.querySelectorAll('input[name="cote_poignee_porte"]').forEach(input => {
        input.addEventListener('change', updateStandPreview);
    });

    // Validation du formulaire
    form.addEventListener('submit', function(event) {
        if (!form.checkValidity()) {
            event.preventDefault();
            event.stopPropagation();
        }
        
        // Ajouter le champ is_custom au formulaire avant l'envoi
        const isCustomInput = document.createElement('input');
        isCustomInput.type = 'hidden';
        isCustomInput.name = 'is_custom';
        isCustomInput.value = document.getElementById('type_custom').checked ? '1' : '0';
        form.appendChild(isCustomInput);
        
        form.classList.add('was-validated');
    });

    // Script pour créer rapidement un client
    document.getElementById('saveQuickClient').addEventListener('click', function() {
        const societe = document.getElementById('new_client_societe').value;
        const secteur = document.getElementById('new_client_secteur').value;
        const nom = document.getElementById('new_client_nom').value;
        const prenom = document.getElementById('new_client_prenom').value;
        const email = document.getElementById('new_client_email').value;
        
        if (!societe) {
            alert('Le nom de la société est requis');
            return;
        }
        
        // Créer le client via une requête AJAX
        const formData = new FormData();
        formData.append('societe', societe);
        formData.append('secteur_activite', secteur);
        formData.append('contact_commercial_nom', nom);
        formData.append('contact_commercial_prenom', prenom);
        formData.append('contact_commercial_mail', email || 'contact@' + societe.toLowerCase().replace(/\s+/g, '') + '.com');
        formData.append('statut', 'prospect');
        
        fetch('../clients/quick_create.php', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Ajouter le nouveau client à la liste
                const select = document.getElementById('client_id');
                const option = document.createElement('option');
                option.value = data.client_id;
                option.textContent = societe + (nom ? ' - ' + prenom + ' ' + nom : '');
                option.selected = true;
                select.appendChild(option);
                
                // Fermer la modal
                bootstrap.Modal.getInstance(document.getElementById('newClientModal')).hide();
                
                // Reset du form
                document.getElementById('quickClientForm').reset();
                
                alert('Client créé avec succès');
                updateStandPreview();
            } else {
                alert('Erreur: ' + data.message);
            }
        })
        .catch(error => {
            alert('Erreur de communication: ' + error.message);
        });
    });

    // Initialiser l'affichage au chargemen
    toggleCreationType(false);
    updateStandPreview();
});
</script>

<?php require_once '../includes/footer.php'; ?>