﻿<?php
require_once '../includes/init.php';
requireLogin();

// Récupération des paramètres
$action = $_GET['action'] ?? 'details';
$id = intval($_GET['id'] ?? 0);

// Vérification de l'existence du stand avec informations client ET NOUVEAUX CHAMPS
try {
    $stmt = $db->prepare("
        SELECT s.*, 
            c.societe as client_nom,
            c.contact_commercial_nom,
            c.contact_commercial_prenom,
            c.contact_commercial_mail,
            c.contact_commercial_tel,
            c.secteur_activite,
            c.pays,
            c.statut as client_statut,
            (
                CASE 
                    WHEN s.is_custom = 1 THEN (
                        SELECT COALESCE(SUM(sp.quantite), 0)
                        FROM stands_pieces sp
                        WHERE sp.stand_id = s.id
                    )
                    ELSE (
                        SELECT COALESCE(SUM(stp.quantite), 0)
                        FROM stand_type_pieces stp 
                        WHERE stp.stand_id = s.type_id
                    ) 
                END
            ) + (
                -- Pièces des options
                SELECT COALESCE(SUM(sop.quantite * eso.quantite), 0)
                FROM event_stands es
                JOIN event_stand_options eso ON eso.event_stand_id = es.id
                JOIN stand_option_pieces sop ON sop.option_id = eso.option_id
                WHERE es.stand_id = s.id
            ) as nb_pieces,
            (SELECT COUNT(*) FROM event_stands WHERE stand_id = s.id) as nb_locations,
            st.code as type_code,
            st.nom as type_nom,
            st.surface as type_surface,
            -- Informations événement associé
            e.nom as event_nom,
            e.lieu as event_lieu,
            e.date_debut as event_date_debut,
            e.date_fin as event_date_fin,
            e.statut as event_statut
        FROM stands s
        LEFT JOIN clients c ON c.id = s.client_id
        LEFT JOIN stand_types st ON st.id = s.type_id
        LEFT JOIN event_stands es ON es.stand_id = s.id
        LEFT JOIN events e ON e.id = es.event_id
        WHERE s.id = ?
        GROUP BY s.id
    ");
    $stmt->execute([$id]);
    $stand = $stmt->fetch();

    if (!$stand) {
        addMessage("Stand non trouvé", "error");
        redirect('liste.php');
    }

    // Récupération des clients pour le formulaire de modification
    $stmt = $db->query("
        SELECT id, societe, contact_commercial_nom, contact_commercial_prenom
        FROM clients 
        WHERE statut IN ('prospect', 'client')
        ORDER BY societe
    ");
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Si on est sur l'action details ou pieces, récupérons aussi les pièces
    if (in_array($action, ['details', 'pieces'])) {
        try {
            if ($stand['is_custom']) {
                // Pour les stands libres
                $stmt = $db->prepare("
                    SELECT 
                        p.*,
                        sp.quantite,
                        sp.piece_id,
                        cp.nom as categorie_nom
                    FROM pieces p
                    JOIN stands_pieces sp ON p.id = sp.piece_id
                    LEFT JOIN categories_pieces cp ON cp.code = p.categorie
                    WHERE sp.stand_id = ?
                    ORDER BY cp.ordre, p.reference
                ");
                $stmt->execute([$stand['id']]);
                $pieces = $stmt->fetchAll();
            } else {
                // Pour les stands préconfigurés
                $stmt = $db->prepare("
                    SELECT p.*, stp.quantite,
                           cp.nom as categorie_nom
                    FROM pieces p
                    JOIN stand_type_pieces stp ON stp.piece_id = p.id
                    LEFT JOIN categories_pieces cp ON cp.code = p.categorie
                    WHERE stp.stand_id = ?
                    ORDER BY cp.ordre, p.reference
                ");
                $stmt->execute([$stand['type_id']]);
                $pieces = $stmt->fetchAll();
            }

            // Récupération des options du stand et leurs pièces
            $stmt = $db->prepare("
                SELECT so.*, eso.quantite,
                       (SELECT COUNT(*) FROM stand_option_pieces WHERE option_id = so.id) as nb_pieces
                FROM event_stand_options eso
                JOIN stand_options so ON so.id = eso.option_id
                JOIN event_stands es ON es.id = eso.event_stand_id
                WHERE es.stand_id = ?
            ");
            $stmt->execute([$id]);
            $options = $stmt->fetchAll();

            $options_pieces = [];
            if (!empty($options)) {
                $stmt = $db->prepare("
                    SELECT p.*, sop.quantite, so.nom as option_nom,
                           cp.nom as categorie_nom
                    FROM stand_option_pieces sop
                    JOIN pieces p ON p.id = sop.piece_id
                    JOIN stand_options so ON so.id = sop.option_id
                    LEFT JOIN categories_pieces cp ON cp.code = p.categorie
                    WHERE sop.option_id = ?
                    ORDER BY cp.ordre, p.reference
                ");

                foreach ($options as $option) {
                    $stmt->execute([$option['id']]);
                    $options_pieces[$option['id']] = $stmt->fetchAll();
                }
            }

        } catch(PDOException $e) {
            error_log("Erreur récupération pièces stand: " . $e->getMessage());
            $pieces = [];
            $options = [];
            $options_pieces = [];
        }
    }
} catch(PDOException $e) {
    error_log("Erreur lors de la récupération du stand : " . $e->getMessage());
    addMessage("Erreur lors de la récupération du stand", "error");
    redirect('liste.php');
}

// Traitement des actions POST - TYPE NON MODIFIABLE
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        switch($action) {
            case 'modifier':
                $db->beginTransaction();
                try {
                    // Validation et nettoyage des données
                    $nom = cleanInput($_POST['nom'] ?? '');
                    $client_id = intval($_POST['client_id'] ?? 0);
                    $description = cleanInput($_POST['description'] ?? '');
                    $surface = floatval($_POST['surface'] ?? 0);
                    $nb_angles = intval($_POST['nb_angles'] ?? 0);
                    $prix_location_ht = floatval($_POST['prix_location_ht'] ?? 0);
                    
                    // NOUVEAUX CHAMPS TECHNIQUES
                    $code_porte = cleanInput($_POST['code_porte'] ?? '');
                    $cote_poignee_porte = cleanInput($_POST['cote_poignee_porte'] ?? '');
                    $nb_plaques_pvc_supplementaires = intval($_POST['nb_plaques_pvc_supplementaires'] ?? 0);
                    $remarques = cleanInput($_POST['remarques'] ?? '');

                    // LE TYPE ET IS_CUSTOM NE SONT PAS MODIFIABLES
                    // On garde les valeurs actuelles du stand

                    // Règles de validation
                    if (empty($nom)) {
                        throw new Exception("Le nom du stand est requis");
                    }
                    if ($client_id <= 0) {
                        throw new Exception("Le client est requis");
                    }
                    
                    // Vérifier que le client existe
                    $stmt = $db->prepare("SELECT id FROM clients WHERE id = ?");
                    $stmt->execute([$client_id]);
                    if (!$stmt->fetch()) {
                        throw new Exception("Le client sélectionné n'existe pas");
                    }
                    
                    if ($surface <= 0) {
                        throw new Exception("La surface doit être supérieure à 0");
                    }
                    if ($prix_location_ht < 0) {
                        throw new Exception("Le prix de location doit être positif");
                    }
                    
                    // VALIDATIONS NOUVEAUX CHAMPS
                    if (!empty($code_porte) && !preg_match('/^[0-9]{3}$/', $code_porte)) {
                        throw new Exception("Le code porte doit contenir exactement 3 chiffres");
                    }
                    if (!empty($cote_poignee_porte) && !in_array($cote_poignee_porte, ['droite', 'gauche'])) {
                        throw new Exception("Le côté de la poignée doit être 'droite' ou 'gauche'");
                    }
                    if ($nb_plaques_pvc_supplementaires < 0) {
                        throw new Exception("Le nombre de plaques PVC doit être positif ou nul");
                    }

                    // Traitement de l'image
                    $image = $stand['image'];
                    if (!empty($_FILES['image']['tmp_name'])) {
                        $upload_dir = '../assets/uploads/stands/';
                        if (!is_dir($upload_dir)) {
                            mkdir($upload_dir, 0755, true);
                        }
                        
                        $file_extension = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));
                        $new_filename = 'stand_' . $id . '_' . time() . '.' . $file_extension;
                        
                        if (!in_array($file_extension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                            throw new Exception("Format d'image non supporté");
                        }
                        
                        if ($_FILES['image']['size'] > 5 * 1024 * 1024) {
                            throw new Exception("L'image est trop volumineuse (max 5MB)");
                        }
                        
                        if (move_uploaded_file($_FILES['image']['tmp_name'], $upload_dir . $new_filename)) {
                            // Suppression de l'ancienne image
                            if ($stand['image'] && file_exists($upload_dir . $stand['image'])) {
                                unlink($upload_dir . $stand['image']);
                            }
                            $image = $new_filename;
                        }
                    }

                    // Mise à jour en base SANS MODIFIER type_id et is_custom
                    $stmt = $db->prepare("
                        UPDATE stands 
                        SET nom = :nom,
                            client_id = :client_id,
                            description = :description, 
                            surface = :surface,
                            nb_angles = :nb_angles, 
                            prix_location_ht = :prix_location_ht, 
                            image = :image,
                            code_porte = :code_porte,
                            cote_poignee_porte = :cote_poignee_porte,
                            nb_plaques_pvc_supplementaires = :nb_plaques_pvc_supplementaires,
                            remarques = :remarques,
                            updated_at = NOW()
                        WHERE id = :id
                    ");

                    $result = $stmt->execute([
                        ':nom' => $nom,
                        ':client_id' => $client_id,
                        ':description' => $description,
                        ':surface' => $surface,
                        ':nb_angles' => $nb_angles,
                        ':prix_location_ht' => $prix_location_ht,
                        ':image' => $image,
                        ':code_porte' => !empty($code_porte) ? $code_porte : null,
                        ':cote_poignee_porte' => !empty($cote_poignee_porte) ? $cote_poignee_porte : null,
                        ':nb_plaques_pvc_supplementaires' => $nb_plaques_pvc_supplementaires,
                        ':remarques' => !empty($remarques) ? $remarques : null,
                        ':id' => $id
                    ]);

                    if (!$result) {
                        throw new Exception("Erreur lors de la mise à jour en base de données");
                    }

                    // Log de l'activité AVEC DÉTAILS TECHNIQUES
                    $stmt = $db->prepare("
                        INSERT INTO activity_logs (
                            user_id, action, details, created_at
                        ) VALUES (?, 'UPDATE', ?, NOW())
                    ");
                    
                    $technicalInfo = "";
                    if (!empty($code_porte)) $technicalInfo .= " - Code porte: " . $code_porte;
                    if (!empty($cote_poignee_porte)) $technicalInfo .= " - Poignée: " . $cote_poignee_porte;
                    if ($nb_plaques_pvc_supplementaires > 0) $technicalInfo .= " - " . $nb_plaques_pvc_supplementaires . " plaque(s) PVC sup.";
                    
                    $logDetails = "Modification du stand ID: $id - Nom: $nom" . $technicalInfo;
                    $stmt->execute([$_SESSION['user_id'], $logDetails]);

                    $db->commit();
                    addMessage("Stand modifié avec succès", "success");
                    redirect("stand/gestion_stand.php?action=details&id=" . $id);

                } catch (Exception $e) {
                    $db->rollBack();
                    error_log("Erreur modification stand " . $id . ": " . $e->getMessage());
                    addMessage("Erreur lors de la modification : " . $e->getMessage(), "error");
                }
                break;
        }
    } catch(Exception $e) {
        if (isset($db) && $db->inTransaction()) {
            $db->rollBack();
        }
        error_log("Erreur générale : " . $e->getMessage());
        addMessage("Erreur : " . $e->getMessage(), "error");
    }
}

// Définition du titre de la page selon l'action
switch($action) {
    case 'details':
        $page_title = "Stand : " . $stand['nom'];
        break;
    case 'modifier':
        $page_title = "Modifier : " . $stand['nom'];
        break;
    case 'pieces':
        $page_title = "Pièces : " . $stand['nom'];
        break;
}

// Fonction pour obtenir la couleur du badge de statut
function getStatusBadge($statut) {
    switch($statut) {
        case 'client': return 'bg-success';
        case 'prospect': return 'bg-warning';
        case 'ancien_client': return 'bg-secondary';
        default: return 'bg-secondary';
    }
}

// Inclusion du header
require_once '../includes/header.php';
?>

<div class="container my-4 fade-in">
    <!-- En-tête moderne avec hero section -->
    <div class="hero-section bg-gradient-primary text-white rounded-4 p-4 mb-4">
        <div class="row align-items-center">
            <div class="col-md-8">
                <div class="d-flex align-items-center mb-3">
                    <div class="icon-box bg-white bg-opacity-20 me-3">
                        <i class="fas fa-store fa-2x text-white"></i>
                    </div>
                    <div>
                        <h1 class="mb-0 text-white"><?= htmlspecialchars($stand['nom']) ?></h1>
                        <?php if ($stand['client_nom']): ?>
                            <p class="mb-0 text-white-50">
                                <i class="fas fa-building me-2"></i><?= htmlspecialchars($stand['client_nom']) ?>
                                <?php if ($stand['client_statut']): ?>
                                    <span class="badge <?= getStatusBadge($stand['client_statut']) ?> ms-2">
                                        <?= ucfirst($stand['client_statut']) ?>
                                    </span>
                                <?php endif; ?>
                            </p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Informations rapides AVEC CODE PORTE -->
                <div class="row g-3">
                    <div class="col-sm-6 col-md-3">
                        <div class="stat-mini">
                            <div class="stat-value"><?= $stand['surface'] ?>m²</div>
                            <div class="stat-label">Surface</div>
                        </div>
                    </div>
                    <div class="col-sm-6 col-md-3">
                        <div class="stat-mini">
                            <div class="stat-value"><?= $stand['nb_pieces'] ?></div>
                            <div class="stat-label">Pièces</div>
                        </div>
                    </div>
                    <div class="col-sm-6 col-md-3">
                        <div class="stat-mini">
                            <div class="stat-value">
                                <?php if (!empty($stand['code_porte'])): ?>
                                    <i class="fas fa-lock me-1"></i><?= htmlspecialchars($stand['code_porte']) ?>
                                <?php else: ?>
                                    <i class="fas fa-lock-open"></i>---
                                <?php endif; ?>
                            </div>
                            <div class="stat-label">Code Porte</div>
                        </div>
                    </div>
                    <div class="col-sm-6 col-md-3">
                        <div class="stat-mini">
                            <div class="stat-value"><?= number_format($stand['prix_location_ht'], 0) ?>€</div>
                            <div class="stat-label">Prix HT</div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-md-4">
                <?php if ($stand['image']): ?>
                    <img src="../assets/uploads/stands/<?= htmlspecialchars($stand['image']) ?>" 
                         alt="<?= htmlspecialchars($stand['nom']) ?>"
                         class="img-fluid rounded-3 shadow">
                <?php else: ?>
                    <div class="text-center p-4 bg-white bg-opacity-10 rounded-3">
                        <i class="fas fa-image fa-3x text-white-50"></i>
                        <p class="text-white-50 mt-2 mb-0">Aucune image</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Navigation par onglets -->
    <div class="card shadow-sm mb-4">
        <div class="card-header bg-white border-bottom-0">
            <ul class="nav nav-tabs card-header-tabs" role="tablist">
                <li class="nav-item">
                    <a class="nav-link <?= $action === 'details' ? 'active' : '' ?>" 
                       href="?action=details&id=<?= $stand['id'] ?>">
                        <i class="fas fa-info-circle me-2"></i>Détails
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?= $action === 'modifier' ? 'active' : '' ?>" 
                       href="?action=modifier&id=<?= $stand['id'] ?>">
                        <i class="fas fa-edit me-2"></i>Modifier
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link <?= $action === 'pieces' ? 'active' : '' ?>" 
                       href="?action=pieces&id=<?= $stand['id'] ?>">
                        <i class="fas fa-puzzle-piece me-2"></i>Pièces 
                        <span class="badge bg-primary ms-1"><?= $stand['nb_pieces'] ?></span>
                    </a>
                </li>
            </ul>
        </div>
    </div>

    <!-- Actions rapides -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div class="btn-group" role="group">
            <a href="generate_stand_inventory.php?id=<?= $stand['id'] ?>" class="btn btn-primary">
                <i class="fas fa-file-pdf me-2"></i>Inventaire PDF
            </a>
            <?php if ($stand['is_custom'] && $action === 'pieces'): ?>
                <a href="select-pieces.php?id=<?= $stand['id'] ?>" class="btn btn-success">
                    <i class="fas fa-plus-circle me-2"></i>Ajouter pièces
                </a>
            <?php endif; ?>
        </div>
        
        <a href="liste.php" class="btn btn-outline-secondary">
            <i class="fas fa-arrow-left me-2"></i>Retour à la liste
        </a>
    </div>

    <?php if ($action === 'details'): ?>
        <!-- Vue détails COMPLÈTE -->
        <div class="row g-4">
            <!-- Informations client -->
            <?php if ($stand['client_nom']): ?>
                <div class="col-md-6">
                    <div class="card h-100 shadow-sm">
                        <div class="card-header bg-light">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-user-tie me-2 text-primary"></i>
                                Informations Client
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="client-info">
                                <h6 class="text-primary"><?= htmlspecialchars($stand['client_nom']) ?></h6>
                                
                                <?php if ($stand['contact_commercial_prenom'] || $stand['contact_commercial_nom']): ?>
                                    <div class="info-item">
                                        <strong>Contact :</strong>
                                        <?= htmlspecialchars($stand['contact_commercial_prenom']) ?> <?= htmlspecialchars($stand['contact_commercial_nom']) ?>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($stand['contact_commercial_mail']): ?>
                                    <div class="info-item">
                                        <strong>Email :</strong>
                                        <a href="mailto:<?= htmlspecialchars($stand['contact_commercial_mail']) ?>">
                                            <?= htmlspecialchars($stand['contact_commercial_mail']) ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($stand['contact_commercial_tel']): ?>
                                    <div class="info-item">
                                        <strong>Téléphone :</strong>
                                        <a href="tel:<?= htmlspecialchars($stand['contact_commercial_tel']) ?>">
                                            <?= htmlspecialchars($stand['contact_commercial_tel']) ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($stand['secteur_activite']): ?>
                                    <div class="info-item">
                                        <strong>Secteur :</strong>
                                        <?= htmlspecialchars($stand['secteur_activite']) ?>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($stand['pays']): ?>
                                    <div class="info-item">
                                        <strong>Pays :</strong>
                                        <?= htmlspecialchars($stand['pays']) ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Informations techniques COMPLÈTES -->
            <div class="col-md-6">
                <div class="card h-100 shadow-sm">
                    <div class="card-header bg-light">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-cog me-2 text-primary"></i>
                            Spécifications Techniques
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="specs-grid">
                            <div class="spec-item">
                                <div class="spec-label">Type</div>
                                <div class="spec-value">
                                    <span class="badge <?= $stand['is_custom'] ? 'bg-info' : 'bg-success' ?>">
                                        <?= $stand['is_custom'] ? 'Personnalisé' : 'Pré-configuré' ?>
                                    </span>
                                    <?php if ($stand['type_nom']): ?>
                                        <span class="ms-2"><?= htmlspecialchars($stand['type_nom']) ?></span>
                                    <?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="spec-item">
                                <div class="spec-label">Surface</div>
                                <div class="spec-value"><?= $stand['surface'] ?> m²</div>
                            </div>
                            
                            <div class="spec-item">
                                <div class="spec-label">Nombre d'angles</div>
                                <div class="spec-value"><?= $stand['nb_angles'] ?></div>
                            </div>

                            <!-- NOUVEAUX CHAMPS TECHNIQUES -->
                            <?php if (!empty($stand['code_porte'])): ?>
                            <div class="spec-item">
                                <div class="spec-label">
                                    <i class="fas fa-lock me-2 text-secondary"></i>Code porte
                                </div>
                                <div class="spec-value">
                                    <code class="bg-light px-2 py-1 rounded text-primary fw-bold"><?= htmlspecialchars($stand['code_porte']) ?></code>
                                </div>
                            </div>
                            <?php endif; ?>

                            <?php if (!empty($stand['cote_poignee_porte'])): ?>
                            <div class="spec-item">
                                <div class="spec-label">
                                    <i class="fas fa-door-open me-2 text-secondary"></i>Côté poignée
                                </div>
                                <div class="spec-value">
                                    <span class="badge bg-outline-secondary border">
                                        <i class="fas fa-arrow-<?= $stand['cote_poignee_porte'] === 'droite' ? 'right' : 'left' ?> me-1"></i>
                                        <?= ucfirst($stand['cote_poignee_porte']) ?>
                                    </span>
                                </div>
                            </div>
                            <?php endif; ?>

                            <?php if ($stand['nb_plaques_pvc_supplementaires'] > 0): ?>
                            <div class="spec-item">
                                <div class="spec-label">
                                    <i class="fas fa-square me-2 text-secondary"></i>Plaques PVC sup.
                                </div>
                                <div class="spec-value">
                                    <span class="badge bg-warning text-dark">
                                        <?= $stand['nb_plaques_pvc_supplementaires'] ?> plaque(s)
                                    </span>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <div class="spec-item">
                                <div class="spec-label">Prix location HT</div>
                                <div class="spec-value text-success fw-bold">
                                    <?= number_format($stand['prix_location_ht'], 2, ',', ' ') ?> €
                                </div>
                            </div>
                            
                            <div class="spec-item">
                                <div class="spec-label">Nombre de pièces</div>
                                <div class="spec-value">
                                    <span class="badge bg-primary"><?= $stand['nb_pieces'] ?></span>
                                </div>
                            </div>
                            
                            <div class="spec-item">
                                <div class="spec-label">Locations</div>
                                <div class="spec-value">
                                    <span class="badge bg-secondary"><?= $stand['nb_locations'] ?></span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- REMARQUES TECHNIQUES (Section dédiée) -->
            <?php if (!empty($stand['remarques'])): ?>
            <div class="col-12">
                <div class="card shadow-sm border-warning">
                    <div class="card-header bg-warning bg-opacity-10">
                        <h5 class="card-title mb-0 text-warning-emphasis">
                            <i class="fas fa-sticky-note me-2"></i>
                            Remarques Techniques
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="alert alert-warning border-0 mb-2">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Pour l'équipe de montage :</strong>
                        </div>
                        <p class="mb-0 fst-italic"><?= nl2br(htmlspecialchars($stand['remarques'])) ?></p>
                    </div>
                </div>
            </div>
            <?php endif; ?>

            <!-- Événement associé -->
            <?php if ($stand['event_nom']): ?>
                <div class="col-12">
                    <div class="card shadow-sm">
                        <div class="card-header bg-light">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-calendar-alt me-2 text-primary"></i>
                                Événement Associé
                            </h5>
                        </div>
                        <div class="card-body">
                            <div class="row align-items-center">
                                <div class="col-md-6">
                                    <h6 class="text-primary mb-2"><?= htmlspecialchars($stand['event_nom']) ?></h6>
                                    <?php if ($stand['event_lieu']): ?>
                                        <p class="text-muted mb-1">
                                            <i class="fas fa-map-marker-alt me-2"></i>
                                            <?= htmlspecialchars($stand['event_lieu']) ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                                <div class="col-md-6">
                                    <div class="text-md-end">
                                        <?php if ($stand['event_date_debut'] && $stand['event_date_fin']): ?>
                                            <div class="event-dates mb-2">
                                                <i class="fas fa-calendar me-2"></i>
                                                <?= date('d/m/Y', strtotime($stand['event_date_debut'])) ?>
                                                -
                                                <?= date('d/m/Y', strtotime($stand['event_date_fin'])) ?>
                                            </div>
                                        <?php endif; ?>
                                        <?php if ($stand['event_statut']): ?>
                                            <span class="badge <?= $stand['event_statut'] === 'preparation' ? 'bg-warning' : 'bg-success' ?>">
                                                <?= ucfirst($stand['event_statut']) ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Description -->
            <?php if ($stand['description']): ?>
                <div class="col-12">
                    <div class="card shadow-sm">
                        <div class="card-header bg-light">
                            <h5 class="card-title mb-0">
                                <i class="fas fa-align-left me-2 text-primary"></i>
                                Description
                            </h5>
                        </div>
                        <div class="card-body">
                            <p class="mb-0"><?= nl2br(htmlspecialchars($stand['description'])) ?></p>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>

    <?php elseif ($action === 'modifier'): ?>
        <!-- Formulaire de modification SANS TYPE MODIFIABLE -->
        <div class="card shadow-sm">
            <div class="card-body">
                <form method="POST" enctype="multipart/form-data" class="needs-validation" novalidate>
                    <div class="row g-4">
                        <!-- SECTION INFORMATIONS GÉNÉRALES -->
                        <div class="col-12">
                            <h6 class="text-primary border-bottom pb-2 mb-3">
                                <i class="fas fa-info-circle me-2"></i>Informations générales
                            </h6>
                        </div>
                        
                        <div class="col-md-6">
                            <label for="client_id" class="form-label">Client *</label>
                            <select class="form-select" id="client_id" name="client_id" required>
                                <option value="">Sélectionner un client</option>
                                <?php foreach ($clients as $client): ?>
                                    <option value="<?= $client['id'] ?>" 
                                            <?= $stand['client_id'] == $client['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($client['societe']) ?>
                                        <?php if ($client['contact_commercial_nom']): ?>
                                            - <?= htmlspecialchars($client['contact_commercial_prenom']) ?> <?= htmlspecialchars($client['contact_commercial_nom']) ?>
                                        <?php endif; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="col-md-6">
                            <label for="nom" class="form-label">Nom du stand *</label>
                            <input type="text" class="form-control" id="nom" name="nom" 
                                   value="<?= htmlspecialchars($stand['nom']) ?>" required>
                        </div>

                        <!-- AFFICHAGE TYPE NON MODIFIABLE -->
                        <div class="col-12">
                            <div class="alert alert-info border-info">
                                <div class="row align-items-center">
                                    <div class="col-md-6">
                                        <h6 class="mb-1">
                                            <i class="fas fa-info-circle me-2"></i>Type de stand (non modifiable)
                                        </h6>
                                        <span class="badge <?= $stand['is_custom'] ? 'bg-warning' : 'bg-success' ?> fs-6">
                                            <?php if ($stand['is_custom']): ?>
                                                <i class="fas fa-cog me-1"></i>Stand Personnalisé
                                            <?php else: ?>
                                                <i class="fas fa-shapes me-1"></i><?= htmlspecialchars($stand['type_nom']) ?>
                                                <?php if ($stand['type_surface']): ?>
                                                    (<?= $stand['type_surface'] ?>m²)
                                                <?php endif; ?>
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                    <div class="col-md-6">
                                        <small class="text-muted">
                                            <i class="fas fa-lock me-1"></i>
                                            Le type ne peut pas être modifié après création. 
                                            Il faut créer un nouveau stand pour changer de type.
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <label for="surface" class="form-label">Surface (m²) *</label>
                            <input type="number" class="form-control" id="surface" name="surface" 
                                   value="<?= $stand['surface'] ?>" required min="1">
                        </div>
                        
                        <div class="col-md-4">
                            <label for="nb_angles" class="form-label">Nombre d'angles</label>
                            <input type="number" class="form-control" id="nb_angles" name="nb_angles" 
                                   value="<?= $stand['nb_angles'] ?>" min="0">
                        </div>
                        
                        <div class="col-md-4">
                            <label for="prix_location_ht" class="form-label">Prix location HT *</label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="prix_location_ht" 
                                       name="prix_location_ht" value="<?= $stand['prix_location_ht'] ?>" 
                                       required step="0.01" min="0">
                                <span class="input-group-text">€</span>
                            </div>
                        </div>

                        <!-- SECTION SPÉCIFICATIONS TECHNIQUES -->
                        <div class="col-12">
                            <h6 class="text-success border-bottom pb-2 mb-3 mt-4">
                                <i class="fas fa-cogs me-2"></i>Spécifications techniques
                            </h6>
                        </div>

                        <!-- Code porte -->
                        <div class="col-md-3">
                            <label for="code_porte" class="form-label">
                                <i class="fas fa-lock me-2 text-secondary"></i>Code porte (3 chiffres)
                            </label>
                            <input type="text" class="form-control" id="code_porte" name="code_porte" 
                                   value="<?= htmlspecialchars($stand['code_porte'] ?? '') ?>" 
                                   maxlength="3" pattern="[0-9]{3}" placeholder="123">
                            <small class="text-muted">Code pour ouvrir la serrure</small>
                        </div>

                        <!-- Côté poignée -->
                        <div class="col-md-3">
                            <label class="form-label">
                                <i class="fas fa-door-open me-2 text-secondary"></i>Côté de la poignée
                            </label>
                            <div class="d-flex gap-2">
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="cote_poignee_porte" 
                                           id="poignee_droite" value="droite" 
                                           <?= $stand['cote_poignee_porte'] === 'droite' ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="poignee_droite">
                                        <i class="fas fa-arrow-right me-1"></i>Droite
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="cote_poignee_porte" 
                                           id="poignee_gauche" value="gauche"
                                           <?= $stand['cote_poignee_porte'] === 'gauche' ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="poignee_gauche">
                                        <i class="fas fa-arrow-left me-1"></i>Gauche
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="cote_poignee_porte" 
                                           id="poignee_non_defini" value=""
                                           <?= empty($stand['cote_poignee_porte']) ? 'checked' : '' ?>>
                                    <label class="form-check-label" for="poignee_non_defini">
                                        <i class="fas fa-times me-1"></i>Non défini
                                    </label>
                                </div>
                            </div>
                            <small class="text-muted">Position vue de l'extérieur</small>
                        </div>

                        <!-- Plaques PVC supplémentaires -->
                        <div class="col-md-3">
                            <label for="nb_plaques_pvc_supplementaires" class="form-label">
                                <i class="fas fa-square me-2 text-secondary"></i>Plaques PVC sup.
                            </label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="nb_plaques_pvc_supplementaires" 
                                       name="nb_plaques_pvc_supplementaires" 
                                       value="<?= $stand['nb_plaques_pvc_supplementaires'] ?? 0 ?>" 
                                       min="0">
                                <span class="input-group-text">plaques</span>
                            </div>
                            <small class="text-muted">Pour parois communes</small>
                        </div>

                        <!-- Bouton réinitialiser les specs techniques -->
                        <div class="col-md-3">
                            <label class="form-label">&nbsp;</label>
                            <div class="d-grid">
                                <button type="button" class="btn btn-outline-secondary btn-sm" id="resetTechnicalSpecs">
                                    <i class="fas fa-eraser me-2"></i>Réinitialiser
                                </button>
                            </div>
                            <small class="text-muted">Efface les spécifications techniques</small>
                        </div>

                        <!-- Remarques techniques -->
                        <div class="col-12">
                            <label for="remarques" class="form-label">
                                <i class="fas fa-sticky-note me-2 text-warning"></i>Remarques techniques
                            </label>
                            <textarea class="form-control" id="remarques" name="remarques" rows="3"
                                      placeholder="Instructions pour l'équipe de montage, contraintes particulières..."><?= htmlspecialchars($stand['remarques'] ?? '') ?></textarea>
                            <small class="text-muted">Notes visibles par l'équipe de montage</small>
                        </div>

                        <!-- SECTION AUTRES INFORMATIONS -->
                        <div class="col-12">
                            <h6 class="text-info border-bottom pb-2 mb-3 mt-4">
                                <i class="fas fa-edit me-2"></i>Autres informations
                            </h6>
                        </div>
                        
                        <div class="col-12">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3"><?= htmlspecialchars($stand['description']) ?></textarea>
                        </div>
                        
                        <div class="col-md-6">
                            <label for="image" class="form-label">Image</label>
                            <input type="file" class="form-control" id="image" name="image" accept="image/*">
                            <?php if ($stand['image']): ?>
                                <small class="text-success">
                                    <i class="fas fa-check me-1"></i>Image actuelle : <?= htmlspecialchars($stand['image']) ?>
                                </small>
                            <?php endif; ?>
                        </div>

                        <!-- RÉSUMÉ DES MODIFICATIONS -->
                        <div class="col-md-6">
                            <div class="card bg-light h-100">
                                <div class="card-body">
                                    <h6 class="card-title text-primary">
                                        <i class="fas fa-eye me-2"></i>Aperçu des modifications
                                    </h6>
                                    <div id="modification-preview" class="small">
                                        <div class="text-muted">Les modifications apparaîtront ici...</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-12">
                            <hr>
                            <div class="d-flex justify-content-between align-items-center">
                                <div class="text-muted">
                                    <i class="fas fa-info-circle me-2"></i>
                                    Les modifications seront sauvegardées dans l'historique
                                </div>
                                <div class="d-flex gap-2">
                                    <a href="?action=details&id=<?= $stand['id'] ?>" class="btn btn-secondary">
                                        <i class="fas fa-times me-2"></i>Annuler
                                    </a>
                                    <button type="submit" class="btn btn-primary">
                                        <i class="fas fa-save me-2"></i>Enregistrer les modifications
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </div>


<?php elseif ($action === 'pieces'): ?>
    <!-- Gestion des pièces - TABLEAU SIMPLIFIÉ -->
    <div class="row g-4">
        <!-- Pièces principales -->
        <div class="col-12">
            <div class="card shadow-sm">
                <div class="card-header bg-light d-flex justify-content-between align-items-center">
                    <h5 class="card-title mb-0">
                        <i class="fas fa-puzzle-piece me-2 text-primary"></i>
                        Pièces du stand
                    </h5>
                    <span class="badge bg-primary"><?= count($pieces ?? []) ?> pièces</span>
                </div>
                <div class="card-body p-0">
                    <?php if (!empty($pieces)): ?>
                        <div class="table-responsive">
                            <table class="table table-hover mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th style="width: 100px;">IMAGE</th>
                                        <th style="width: 200px;">RÉFÉRENCE</th>
                                        <th>DÉSIGNATION</th>
                                        <th style="width: 150px;">CATÉGORIE</th>
                                        <th style="width: 100px;" class="text-center">NOMBRE</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($pieces as $piece): ?>
                                        <tr>
                                            <td class="align-middle">
                                                <?php if (!empty($piece['image_principale'])): ?>
                                                    <img src="../assets/images/pieces/<?= htmlspecialchars($piece['image_principale']) ?>" 
                                                         alt="<?= htmlspecialchars($piece['designation']) ?>"
                                                         class="img-thumbnail"
                                                         style="width: 60px; height: 60px; object-fit: cover;">
                                                <?php else: ?>
                                                    <div class="bg-light d-flex align-items-center justify-content-center border rounded"
                                                         style="width: 60px; height: 60px;">
                                                        <i class="fas fa-image text-muted"></i>
                                                    </div>
                                                <?php endif; ?>
                                            </td>
                                            <td class="align-middle">
                                                <?= htmlspecialchars($piece['reference']) ?>
                                            </td>
                                            <td class="align-middle">
                                                <?= htmlspecialchars($piece['designation']) ?>
                                            </td>
                                            <td class="align-middle">
                                                <span class="badge bg-secondary">
                                                    <?= htmlspecialchars($piece['categorie_nom'] ?? $piece['categorie']) ?>
                                                </span>
                                            </td>
                                            <td class="text-center align-middle">
                                                <span class="badge bg-primary fs-6 px-3 py-2">
                                                    <?= $piece['quantite'] ?>
                                                </span>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Résumé par catégorie -->
                        <div class="card-footer bg-light">
                            <h6 class="mb-3">Résumé par catégorie</h6>
                            <div class="row">
                                <?php
                                $categoriesCount = [];
                                foreach ($pieces as $piece) {
                                    $cat = $piece['categorie_nom'] ?? $piece['categorie'];
                                    if (!isset($categoriesCount[$cat])) {
                                        $categoriesCount[$cat] = 0;
                                    }
                                    $categoriesCount[$cat] += $piece['quantite'];
                                }
                                ?>
                                <?php foreach ($categoriesCount as $categorie => $total): ?>
                                    <div class="col-md-3 col-sm-6 mb-2">
                                        <div class="text-center p-2 border rounded bg-white">
                                            <div class="fw-bold small"><?= htmlspecialchars($categorie) ?></div>
                                            <span class="badge bg-secondary"><?= $total ?> pièces</span>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>

                    <?php else: ?>
                        <div class="text-center p-5">
                            <i class="fas fa-puzzle-piece fa-4x text-muted mb-3"></i>
                            <h5 class="text-muted">Aucune pièce configurée</h5>
                            <p class="text-muted">Ce stand n'a pas encore de pièces associées</p>
                            <?php if ($stand['is_custom']): ?>
                                <a href="select-pieces.php?id=<?= $stand['id'] ?>" class="btn btn-primary">
                                    <i class="fas fa-plus me-2"></i>Ajouter des pièces
                                </a>
                            <?php endif; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Options du stand -->
        <?php if (!empty($options)): ?>
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-warning bg-opacity-10">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-cog me-2 text-warning"></i>
                            Options du stand
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php foreach ($options as $option): ?>
                            <div class="border-start border-warning border-4 ps-3 mb-4">
                                <div class="d-flex justify-content-between align-items-center mb-3">
                                    <h6 class="text-warning mb-0">
                                        <i class="fas fa-star me-2"></i>
                                        <?= htmlspecialchars($option['nom']) ?>
                                    </h6>
                                    <span class="badge bg-warning text-dark">
                                        Quantité: <?= $option['quantite'] ?>
                                    </span>
                                </div>
                                
                                <?php if (!empty($options_pieces[$option['id']])): ?>
                                    <div class="table-responsive">
                                        <table class="table table-sm">
                                            <thead class="table-light">
                                                <tr>
                                                    <th style="width: 60px;">Image</th>
                                                    <th style="width: 150px;">Référence</th>
                                                    <th>Désignation</th>
                                                    <th style="width: 120px;">Catégorie</th>
                                                    <th style="width: 80px;" class="text-center">Quantité</th>
                                                </tr>
                                            </thead>
                                            <tbody>
                                                <?php foreach ($options_pieces[$option['id']] as $piece): ?>
                                                    <tr>
                                                        <td class="align-middle">
                                                            <?php if (!empty($piece['image_principale'])): ?>
                                                                <img src="../assets/images/pieces/<?= htmlspecialchars($piece['image_principale']) ?>" 
                                                                     alt="<?= htmlspecialchars($piece['designation']) ?>"
                                                                     class="img-thumbnail"
                                                                     style="width: 40px; height: 40px; object-fit: cover;">
                                                            <?php else: ?>
                                                                <div class="bg-light d-flex align-items-center justify-content-center border rounded"
                                                                     style="width: 40px; height: 40px;">
                                                                    <i class="fas fa-image text-muted small"></i>
                                                                </div>
                                                            <?php endif; ?>
                                                        </td>
                                                        <td class="align-middle">
                                                            <small><?= htmlspecialchars($piece['reference']) ?></small>
                                                        </td>
                                                        <td class="align-middle">
                                                            <div class="fw-semibold small">
                                                                <?= htmlspecialchars($piece['designation']) ?>
                                                            </div>
                                                        </td>
                                                        <td class="align-middle">
                                                            <span class="badge bg-light text-dark border">
                                                                <?= htmlspecialchars($piece['categorie_nom'] ?? $piece['categorie']) ?>
                                                            </span>
                                                        </td>
                                                        <td class="text-center align-middle">
                                                            <span class="badge bg-warning text-dark">
                                                                <?= $piece['quantite'] ?>
                                                            </span>
                                                        </td>
                                                    </tr>
                                                <?php endforeach; ?>
                                            </tbody>
                                        </table>
                                    </div>
                                <?php else: ?>
                                    <div class="text-muted small">Aucune pièce définie pour cette option</div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
<?php endif; ?>

<style>
.hero-section {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
}

.stat-mini {
    text-align: center;
}

.stat-value {
    font-size: 1.5rem;
    font-weight: bold;
    color: white;
}

.stat-label {
    font-size: 0.875rem;
    color: rgba(255, 255, 255, 0.8);
}

.client-info .info-item {
    margin-bottom: 0.5rem;
    padding: 0.25rem 0;
    border-bottom: 1px solid #f8f9fa;
}

.client-info .info-item:last-child {
    border-bottom: none;
    margin-bottom: 0;
}

.specs-grid {
    display: grid;
    gap: 1rem;
}

.spec-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 0.5rem 0;
    border-bottom: 1px solid #f8f9fa;
}

.spec-item:last-child {
    border-bottom: none;
}

spec-value {
    font-weight: 600;
}

.event-dates {
    font-size: 0.95rem;
    color: #495057;
}

.fade-in {
    animation: fadeIn 0.5s ease-in;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.nav-tabs .nav-link {
    border: none;
    color: #495057 !important;
    font-weight: 600;
    padding: 1rem 1.5rem;
}

.nav-tabs .nav-link:hover {
    color: #0d6efd !important;
    border-color: transparent;
    background-color: #f8f9fa;
}

.nav-tabs .nav-link.active {
    color: #0d6efd !important;
    background-color: transparent;
    border-bottom: 3px solid #0d6efd;
}

.card {
    transition: box-shadow 0.15s ease-in-out;
}

.card:hover {
    box-shadow: 0 0.5rem 1rem rgba(0, 0, 0, 0.1) !important;
}

.bg-outline-secondary {
    background-color: #f8f9fa !important;
    border: 1px solid #dee2e6 !important;
    color: #495057 !important;
}

#code_porte:invalid {
    border-color: #dc3545;
}

#code_porte:valid:not(:placeholder-shown) {
    border-color: #198754;
}

.alert-warning.border-0 {
    background-color: #fff3cd;
    border-left: 4px solid #ffc107 !important;
}

.text-warning-emphasis {
    color: #664d03 !important;
}

.border-bottom {
    border-bottom: 2px solid #e9ecef !important;
}

code.text-primary {
    font-size: 1rem;
    font-weight: 700;
    background-color: #e7f1ff !important;
    border: 1px solid #0d6efd;
}

.piece-thumb {
    transition: transform 0.2s ease;
    cursor: pointer;
}

.piece-thumb:hover {
    transform: scale(1.05);
}

.option-section {
    border-left: 4px solid #ffc107;
    padding-left: 1rem;
}

.card-sm .card-body {
    padding: 0.75rem;
}

.modal-lg {
    max-width: 800px;
}

.icon-box {
    width: 60px;
    height: 60px;
    border-radius: 15px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.bg-gradient-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%) !important;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const codePorteInput = document.getElementById('code_porte');
    const previewDiv = document.getElementById('modification-preview');
    
    // Validation code porte
    if (codePorteInput) {
        codePorteInput.addEventListener('input', function(e) {
            e.target.value = e.target.value.replace(/[^0-9]/g, '').substring(0, 3);
            updatePreview();
        });
    }

    // Bouton réinitialiser specs techniques
    const resetBtn = document.getElementById('resetTechnicalSpecs');
    if (resetBtn) {
        resetBtn.addEventListener('click', function() {
            document.getElementById('code_porte').value = '';
            document.querySelectorAll('input[name="cote_poignee_porte"]').forEach(radio => {
                if (radio.value === '') radio.checked = true;
                else radio.checked = false;
            });
            document.getElementById('nb_plaques_pvc_supplementaires').value = '0';
            document.getElementById('remarques').value = '';
            updatePreview();
        });
    }

    // Mise à jour aperçu temps réel
    function updatePreview() {
        if (!previewDiv) return;
        
        const nom = document.getElementById('nom')?.value || 'Sans nom';
        const codePorte = document.getElementById('code_porte')?.value || 'Non défini';
        const cotePoignee = document.querySelector('input[name="cote_poignee_porte"]:checked')?.value || 'Non défini';
        const plaquesPvc = document.getElementById('nb_plaques_pvc_supplementaires')?.value || '0';
        const surface = document.getElementById('surface')?.value || '-';
        const prix = document.getElementById('prix_location_ht')?.value || '-';
        
        previewDiv.innerHTML = `
            <div class="row g-2 small">
                <div class="col-6"><strong>Nom:</strong> ${nom}</div>
                <div class="col-6"><strong>Surface:</strong> ${surface} m²</div>
                <div class="col-6"><strong>Prix:</strong> ${prix} € HT</div>
                <div class="col-6"><strong>Code porte:</strong> ${codePorte}</div>
                <div class="col-6"><strong>Poignée:</strong> ${cotePoignee}</div>
                <div class="col-6"><strong>Plaques PVC:</strong> ${plaquesPvc}</div>
            </div>
        `;
    }

    // Écouteurs pour mise à jour aperçu
    ['nom', 'surface', 'prix_location_ht', 'nb_plaques_pvc_supplementaires'].forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            element.addEventListener('input', updatePreview);
            element.addEventListener('change', updatePreview);
        }
    });

    document.querySelectorAll('input[name="cote_poignee_porte"]').forEach(radio => {
        radio.addEventListener('change', updatePreview);
    });

    // Validation du formulaire
    const validationForm = document.querySelector('.needs-validation');
    if (validationForm) {
        validationForm.addEventListener('submit', function(event) {
            if (!validationForm.checkValidity()) {
                event.preventDefault();
                event.stopPropagation();
            }
            validationForm.classList.add('was-validated');
        });
    }

    // Initialiser l'aperçu
    updatePreview();

    // Animation d'entrée pour les modals
    document.querySelectorAll('.modal').forEach(modal => {
        modal.addEventListener('show.bs.modal', function() {
            this.style.animation = 'fadeIn 0.3s ease-in';
        });
    });

    // Tooltips Bootstrap pour les boutons QR Code
    const tooltipTriggerList = [].slice.call(document.querySelectorAll('[title]'));
    tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });

    // Effet hover sur les cartes
    document.querySelectorAll('.card').forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px)';
            this.style.transition = 'transform 0.2s ease';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });

    // Amélioration UX pour les images
    document.querySelectorAll('.piece-thumb').forEach(thumb => {
        thumb.addEventListener('click', function() {
            const modal = this.closest('tr')?.querySelector('[data-bs-toggle="modal"]');
            if (modal) {
                modal.click();
            }
        });
    });

    // Préchargement des images
    document.querySelectorAll('img[src*="pieces/"]').forEach(img => {
        img.addEventListener('error', function() {
            this.style.display = 'none';
            const placeholder = this.nextElementSibling || this.parentNode.querySelector('.bg-light');
            if (placeholder) {
                placeholder.style.display = 'flex';
            }
        });
    });

    // Gestion responsive des tableaux
    function handleResponsive() {
        const tables = document.querySelectorAll('.table-responsive');
        tables.forEach(table => {
            if (window.innerWidth < 768) {
                table.style.fontSize = '0.875rem';
            } else {
                table.style.fontSize = '';
            }
        });
    }

    window.addEventListener('resize', handleResponsive);
    handleResponsive();

    // Amélioration de l'accessibilité
    document.querySelectorAll('button[data-bs-toggle="modal"]').forEach(btn => {
        btn.addEventListener('click', function() {
            const modalId = this.getAttribute('data-bs-target');
            const modal = document.querySelector(modalId);
            if (modal) {
                setTimeout(() => {
                    const closeBtn = modal.querySelector('.btn-close');
                    if (closeBtn) closeBtn.focus();
                }, 300);
            }
        });
    });

    // Raccourcis clavier
    document.addEventListener('keydown', function(e) {
        // Échap pour fermer les modals
        if (e.key === 'Escape') {
            const openModal = document.querySelector('.modal.show');
            if (openModal) {
                const closeBtn = openModal.querySelector('.btn-close');
                if (closeBtn) closeBtn.click();
            }
        }
        
        // Ctrl+S pour sauvegarder (formulaire de modification)
        if (e.ctrlKey && e.key === 's') {
            e.preventDefault();
            const submitBtn = document.querySelector('button[type="submit"]');
            if (submitBtn && submitBtn.closest('form')) {
                submitBtn.click();
            }
        }
    });

    // Messages d'état améliorés
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        if (alert.classList.contains('alert-success')) {
            setTimeout(() => {
                alert.style.opacity = '0';
                alert.style.transition = 'opacity 0.5s ease';
                setTimeout(() => alert.remove(), 500);
            }, 5000);
        }
    });

    // Chargement progressif des images
    const imageObserver = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const img = entry.target;
                if (img.dataset.src) {
                    img.src = img.dataset.src;
                    img.classList.remove('lazy');
                    observer.unobserve(img);
                }
            }
        });
    });

    document.querySelectorAll('img[data-src]').forEach(img => {
        imageObserver.observe(img);
    });

    // Amélioration de l'UX des formulaires
    const inputs = document.querySelectorAll('input, select, textarea');
    inputs.forEach(input => {
        input.addEventListener('focus', function() {
            this.closest('.form-group, .col-md-3, .col-md-4, .col-md-6, .col-12')?.classList.add('focused');
        });
        
        input.addEventListener('blur', function() {
            this.closest('.form-group, .col-md-3, .col-md-4, .col-md-6, .col-12')?.classList.remove('focused');
        });
    });

    // Validation en temps réel du code porte
    if (codePorteInput) {
        codePorteInput.addEventListener('blur', function() {
            const value = this.value;
            const feedback = this.parentNode.querySelector('.invalid-feedback') || 
                           this.parentNode.appendChild(document.createElement('div'));
            feedback.className = 'invalid-feedback';
            
            if (value && !value.match(/^[0-9]{3}$/)) {
                this.classList.add('is-invalid');
                feedback.textContent = 'Le code doit contenir exactement 3 chiffres';
            } else {
                this.classList.remove('is-invalid');
                feedback.textContent = '';
            }
        });
    }

    // Sauvegarde automatique en brouillon (localStorage)
    const postForm = document.querySelector('form[method="POST"]');
    if (postForm && window.location.search.includes('action=modifier')) {
        const standId = new URLSearchParams(window.location.search).get('id');
        const draftKey = `stand_draft_${standId}`;
        
        // Charger le brouillon
        const savedDraft = localStorage.getItem(draftKey);
        if (savedDraft) {
            try {
                const draft = JSON.parse(savedDraft);
                Object.keys(draft).forEach(key => {
                    const input = postForm.querySelector(`[name="${key}"]`);
                    if (input && input.type !== 'file') {
                        if (input.type === 'radio') {
                            const radio = postForm.querySelector(`[name="${key}"][value="${draft[key]}"]`);
                            if (radio) radio.checked = true;
                        } else {
                            input.value = draft[key];
                        }
                    }
                });
                updatePreview();
            } catch (e) {
                console.warn('Erreur lors du chargement du brouillon:', e);
            }
        }
        
        // Sauvegarder périodiquement
        setInterval(() => {
            const formData = new FormData(postForm);
            const draft = {};
            for (let [key, value] of formData.entries()) {
                if (key !== 'image') { // Exclure les fichiers
                    draft[key] = value;
                }
            }
            localStorage.setItem(draftKey, JSON.stringify(draft));
        }, 30000); // Toutes les 30 secondes
        
        // Nettoyer le brouillon à la soumission
        postForm.addEventListener('submit', () => {
            localStorage.removeItem(draftKey);
        });
    }

    console.log('Gestion Stand - Interface initialisée avec succès');
});

// Fonction utilitaire pour le formatage des nombres
function formatNumber(num, decimals = 2) {
    return new Intl.NumberFormat('fr-FR', {
        minimumFractionDigits: decimals,
        maximumFractionDigits: decimals
    }).format(num);
}

// Fonction pour copier le code porte dans le presse-papiers
function copyCodePorte(code) {
    if (navigator.clipboard) {
        navigator.clipboard.writeText(code).then(() => {
            // Feedback visuel
            const btn = event.target.closest('button') || event.target;
            const originalText = btn.innerHTML;
            btn.innerHTML = '<i class="fas fa-check text-success"></i>';
            setTimeout(() => {
                btn.innerHTML = originalText;
            }, 1500);
        });
    }
}
</script>

<?php require_once '../includes/footer.php'; ?>