﻿<?php
require_once '../includes/init.php';
requireLogin();

try {
    // Récupération des événements avec plus de détails
    $eventsQuery = $db->query("
        SELECT DISTINCT 
            e.id, e.nom, e.date_debut, e.date_fin, e.lieu,
            COUNT(es.stand_id) as nb_stands_assignes
        FROM events e
        JOIN event_stands es ON e.id = es.event_id
        WHERE e.date_fin >= CURDATE()
        GROUP BY e.id
        ORDER BY e.date_debut ASC
    ");
    $events = $eventsQuery->fetchAll();

    // Récupération des clients pour le filtre
    $clientsQuery = $db->query("
        SELECT DISTINCT c.id, c.societe, COUNT(s.id) as nb_stands
        FROM clients c
        LEFT JOIN stands s ON c.id = s.client_id
        WHERE c.statut = 'client_actif'
        GROUP BY c.id
        HAVING nb_stands > 0
        ORDER BY c.societe
    ");
    $clients_filter = $clientsQuery->fetchAll();

    // Liste des stands avec informations client
    $sql = "
        SELECT s.*,
               st.nom as type_nom, 
               st.surface as type_surface,
               c.societe as client_societe,
               c.pays as client_pays,
               c.contact_commercial_prenom,
               c.contact_commercial_nom,
               c.contact_commercial_mail,
               e.id as event_id,
               e.nom as event_nom,
               e.lieu as event_lieu,
               e.date_debut as event_date_debut,
               e.date_fin as event_date_fin,
               CASE 
                   WHEN s.is_custom = 1 THEN (
                       SELECT COALESCE(SUM(sp.quantite), 0)
                       FROM stands_pieces sp
                       WHERE sp.stand_id = s.id
                   )
                   ELSE (
                       SELECT COALESCE(SUM(stp.quantite), 0)
                       FROM stand_type_pieces stp 
                       WHERE stp.stand_id = s.type_id
                   ) 
               END + (
                   SELECT COALESCE(SUM(sop.quantite * eso.quantite), 0)
                   FROM event_stands es
                   JOIN event_stand_options eso ON eso.event_stand_id = es.id
                   JOIN stand_option_pieces sop ON sop.option_id = eso.option_id
                   WHERE es.stand_id = s.id
               ) as nb_pieces,
               s.prix_location_ht + COALESCE(
                   (SELECT SUM(so.prix_location_ht * eso.quantite)
                   FROM event_stands es
                   JOIN event_stand_options eso ON eso.event_stand_id = es.id
                   JOIN stand_options so ON so.id = eso.option_id
                   WHERE es.stand_id = s.id), 0
               ) as prix_total_ht,
               es.statut as event_statut
        FROM stands s
        LEFT JOIN stand_types st ON st.id = s.type_id
        LEFT JOIN clients c ON c.id = s.client_id
        LEFT JOIN event_stands es ON es.stand_id = s.id
        LEFT JOIN events e ON e.id = es.event_id
        ORDER BY s.created_at DESC, s.surface, s.nom
    ";
    
    $stands = $db->query($sql)->fetchAll(PDO::FETCH_ASSOC);

} catch(PDOException $e) {
    addMessage("Erreur lors de la récupération des données : " . $e->getMessage(), "error");
    $stands = [];
    $events = [];
    $clients_filter = [];
}

$page_title = "Gestion des stands BeMatrix";
require_once '../includes/header.php';
?>

<div class="container-fluid my-4 fade-in">
    
    <!-- En-tête avec titre et actions -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="mb-1">
                <i class="fas fa-store me-2 text-primary"></i><?= $page_title ?>
            </h1>
            <p class="text-muted mb-0">
                Sélectionnez un événement pour voir les stands associés et leurs statistiques opérationnelles
            </p>
        </div>
        <div class="btn-group">
            <button type="button" id="viewToggle" class="btn btn-outline-secondary">
                <i class="fas fa-th me-2"></i>Vue grille
            </button>
            <a href="creation.php" class="btn btn-primary">
                <i class="fas fa-plus-circle me-2"></i>Nouveau stand
            </a>
            <div class="dropdown">
                <button class="btn btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                    <i class="fas fa-download me-2"></i>Export
                </button>
                <ul class="dropdown-menu">
                    <li><a class="dropdown-item" href="export.php?format=excel">
                        <i class="fas fa-file-excel text-success me-2"></i>Export Excel
                    </a></li>
                    <li><a class="dropdown-item" href="export.php?format=pdf">
                        <i class="fas fa-file-pdf text-danger me-2"></i>Export PDF
                    </a></li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Sélection d'événement prioritaire -->
    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <div class="row align-items-center">
                <div class="col-md-8">
                    <h5 class="mb-2">
                        <i class="fas fa-calendar-alt text-primary me-2"></i>Sélectionner un événement
                    </h5>
                    <p class="text-muted mb-3">Choisissez un événement pour voir les stands associés et leurs statistiques</p>
                    
                    <div class="row g-3">
                        <div class="col-md-8">
                            <select class="form-select form-select-lg" id="eventFilter">
                                <option value="">-- Sélectionner un événement --</option>
                                <?php foreach($events as $event): ?>
                                    <option value="<?= $event['id'] ?>" 
                                            data-stands="<?= $event['nb_stands_assignes'] ?>"
                                            data-lieu="<?= e($event['lieu']) ?>"
                                            data-debut="<?= $event['date_debut'] ?>"
                                            data-fin="<?= $event['date_fin'] ?>">
                                        <?= e($event['nom']) ?> - <?= e($event['lieu']) ?>
                                        (<?= formatDate($event['date_debut']) ?> → <?= formatDate($event['date_fin']) ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <button type="button" class="btn btn-outline-primary" onclick="showAllStands()">
                                <i class="fas fa-eye me-2"></i>Voir tous les stands
                            </button>
                        </div>
                    </div>
                </div>
                <div class="col-md-4 text-center">
                    <div class="bg-light rounded-3 p-4">
                        <i class="fas fa-info-circle fa-2x text-muted mb-2"></i>
                        <p class="text-muted mb-0 small">
                            Sélectionnez un événement pour accéder aux statistiques détaillées et aux actions spécialisées
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Statistiques de l'événement sélectionné (cachées par défaut) -->
    <div id="eventStats" class="row mb-5 g-4" style="display: none;">
        
        <!-- Informations événement -->
        <div class="col-xl-3 col-md-6">
            <div class="card shadow-sm h-100 border-start border-primary border-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <h6 class="text-muted mb-2 fw-normal">Événement</h6>
                            <h5 class="mb-1" id="eventName">-</h5>
                            <small class="text-muted">
                                <i class="fas fa-map-marker-alt me-1"></i><span id="eventLieu">-</span><br>
                                <i class="fas fa-calendar me-1"></i><span id="eventDates">-</span>
                            </small>
                        </div>
                        <div class="icon-wrapper bg-primary bg-opacity-10 rounded-3 p-3">
                            <i class="fas fa-calendar-alt fa-lg text-primary"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Stands de l'événement -->
        <div class="col-xl-3 col-md-6">
            <div class="card shadow-sm h-100 border-start border-success border-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <h6 class="text-muted mb-2 fw-normal">Stands assignés</h6>
                            <h2 class="mb-1 text-success" id="eventStandsCount">0</h2>
                            <small class="text-muted">
                                <span id="eventSurfaceTotal">0</span>m² total
                                • <span id="eventStandsTypes">-</span>
                            </small>
                        </div>
                        <div class="icon-wrapper bg-success bg-opacity-10 rounded-3 p-3">
                            <i class="fas fa-store fa-lg text-success"></i>
                        </div>
                    </div>
                    <div class="progress mt-3" style="height: 4px;">
                        <div class="progress-bar bg-success" id="eventStandsProgress" style="width: 0%"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Matériel requis -->
        <div class="col-xl-3 col-md-6">
            <div class="card shadow-sm h-100 border-start border-info border-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <h6 class="text-muted mb-2 fw-normal">Pièces nécessaires</h6>
                            <h2 class="mb-1 text-info" id="eventPiecesTotal">0</h2>
                            <small class="text-muted">
                                <span id="eventPiecesStatus">En cours de calcul...</span>
                            </small>
                        </div>
                        <div class="icon-wrapper bg-info bg-opacity-10 rounded-3 p-3">
                            <i class="fas fa-puzzle-piece fa-lg text-info"></i>
                        </div>
                    </div>
                    <div class="progress mt-3" style="height: 4px;">
                        <div class="progress-bar bg-info" id="eventPiecesProgress" style="width: 0%"></div>
                    </div>
                </div>
            </div>
        </div>

        <!-- CA de l'événement -->
        <div class="col-xl-3 col-md-6">
            <div class="card shadow-sm h-100 border-start border-warning border-4">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <h6 class="text-muted mb-2 fw-normal">CA Événement</h6>
                            <h2 class="mb-1 text-warning" id="eventCA">0€</h2>
                            <small class="text-muted">
                                HT • <span id="eventCAMoyenne">0</span>€ par stand
                            </small>
                        </div>
                        <div class="icon-wrapper bg-warning bg-opacity-10 rounded-3 p-3">
                            <i class="fas fa-euro-sign fa-lg text-warning"></i>
                        </div>
                    </div>
                    <div class="progress mt-3" style="height: 4px;">
                        <div class="progress-bar bg-warning" style="width: 75%"></div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filtres avancés -->
    <div class="card shadow-sm mb-4" id="filtersCard">
        <div class="card-header bg-gradient-primary text-white">
            <div class="d-flex justify-content-between align-items-center">
                <h5 class="mb-0">
                    <i class="fas fa-filter me-2"></i>Filtres avancés
                </h5>
                <button class="btn btn-outline-light btn-sm" type="button" data-bs-toggle="collapse" 
                        data-bs-target="#advancedFilters">
                    <i class="fas fa-cogs me-1"></i>Options
                </button>
            </div>
        </div>
        
        <div class="card-body">
            <div class="row g-3 mb-3">
                <!-- Recherche enrichie -->
                <div class="col-md-4">
                    <label class="form-label fw-semibold">Recherche globale</label>
                    <div class="input-group">
                        <span class="input-group-text bg-primary text-white border-primary">
                            <i class="fas fa-search"></i>
                        </span>
                        <input type="text" class="form-control border-primary" id="searchInput" 
                               placeholder="Nom, client, lieu, événement...">
                        <button type="button" class="btn btn-outline-secondary" id="clearSearch">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                    <small class="text-muted">Recherche dans tous les champs</small>
                </div>

                <!-- Filtre client -->
                <div class="col-md-3">
                    <label class="form-label fw-semibold">Client</label>
                    <select class="form-select border-success" id="clientFilter">
                        <option value="">Tous les clients</option>
                        <option value="__non_assigne__">⚠️ Non assignés</option>
                        <?php foreach($clients_filter as $client): ?>
                            <option value="<?= $client['id'] ?>">
                                <?= e($client['societe']) ?> 
                                (<?= $client['nb_stands'] ?> stand<?= $client['nb_stands'] > 1 ? 's' : '' ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <!-- Type de stand -->
                <div class="col-md-3">
                    <label class="form-label fw-semibold">Type</label>
                    <select class="form-select border-info" id="typeFilter">
                        <option value="">Tous types</option>
                        <option value="preconfigure">Pré-configuré</option>
                        <option value="personnalise">Personnalisé</option>
                    </select>
                </div>

                <!-- Actions rapides -->
                <div class="col-md-2">
                    <label class="form-label fw-semibold">Actions</label>
                    <div class="d-grid gap-1">
                        <button type="button" id="resetFilters" class="btn btn-outline-secondary">
                            <i class="fas fa-undo me-1"></i>Reset
                        </button>
                    </div>
                </div>
            </div>

            <!-- Filtres avancés (collapsibles) -->
            <div class="collapse" id="advancedFilters">
                <hr class="my-3">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Surface (m²)</label>
                        <div class="row g-1">
                            <div class="col-6">
                                <input type="number" class="form-control form-control-sm" id="surfaceMin" 
                                       placeholder="Min" min="0" step="0.1">
                            </div>
                            <div class="col-6">
                                <input type="number" class="form-control form-control-sm" id="surfaceMax" 
                                       placeholder="Max" min="0" step="0.1">
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Prix (€ HT)</label>
                        <div class="row g-1">
                            <div class="col-6">
                                <input type="number" class="form-control form-control-sm" id="prixMin" 
                                       placeholder="Min" min="0">
                            </div>
                            <div class="col-6">
                                <input type="number" class="form-control form-control-sm" id="prixMax" 
                                       placeholder="Max" min="0">
                            </div>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Statut</label>
                        <select class="form-select form-select-sm" id="statutFilter">
                            <option value="">Tous statuts</option>
                            <option value="avec_client">Avec client</option>
                            <option value="sans_client">Sans client</option>
                            <option value="avec_evenement">Avec événement</option>
                        </select>
                    </div>
                </div>
            </div>
        </div>

        <!-- Actions spécifiques à l'événement -->
        <div class="card-footer bg-light" id="eventActions" style="display: none;">
            <div class="d-flex justify-content-between align-items-center">
                <div class="text-muted">
                    <span id="resultCounter">Chargement...</span>
                </div>
                <div class="btn-group">
                    <button type="button" 
                            class="btn btn-outline-primary btn-sm" 
                            id="generateInventoryBtn">
                        <i class="fas fa-file-pdf me-1"></i>
                        Générer tous les bons de préparation
                    </button>
                    <button type="button" 
                            class="btn btn-outline-success btn-sm" 
                            id="exportEventBtn">
                        <i class="fas fa-download me-1"></i>
                        Export événement
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Vue Liste -->
    <div id="tableView" class="card shadow-sm">
        <div class="table-responsive">
            <table class="table table-hover align-middle mb-0" id="standsTable">
                <thead class="table-dark">
                    <tr>
                        <th class="ps-3">
                            <input type="checkbox" class="form-check-input" id="selectAll">
                        </th>
                        <th>Image</th>
                        <th class="sortable" data-sort="nom">
                            <i class="fas fa-sort me-1"></i>Stand & Client
                        </th>
                        <th class="sortable" data-sort="categorie">
                            <i class="fas fa-sort me-1"></i>Catégorie
                        </th>
                        <th class="sortable text-center" data-sort="surface">
                            <i class="fas fa-sort me-1"></i>Dimensions
                        </th>
                        <th class="sortable text-center" data-sort="pieces">
                            <i class="fas fa-sort me-1"></i>Pièces
                        </th>
                        <th class="sortable text-end" data-sort="prix">
                            <i class="fas fa-sort me-1"></i>Tarification
                        </th>
                        <th class="text-center">Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($stands as $stand): ?>
                    <tr class="stand-row" 
                        data-event-id="<?= $stand['event_id'] ?? '' ?>" 
                        data-event-name="<?= e($stand['event_nom'] ?? '') ?>"
                        data-client-id="<?= $stand['client_id'] ?? '' ?>"
                        data-client-name="<?= e($stand['client_societe'] ?? '') ?>"
                        data-surface="<?= $stand['surface'] ?? 0 ?>"
                        data-prix="<?= $stand['prix_total_ht'] ?? 0 ?>"
                        data-type="<?= !empty($stand['is_custom']) ? 'personnalise' : 'preconfigure' ?>">
                        <td class="ps-3">
                            <input type="checkbox" class="form-check-input stand-checkbox" 
                                   value="<?= $stand['id'] ?>">
                        </td>
                        <td>
                            <div class="position-relative">
                                <?php if ($stand['image']): ?>
                                    <img src="/gestionstock/assets/uploads/stands/<?= e($stand['image']) ?>" 
                                         alt="<?= e($stand['nom']) ?>"
                                         class="img-thumbnail shadow-sm"
                                         style="max-width: 80px; max-height: 80px;">
                                <?php else: ?>
                                    <div class="bg-light rounded text-center d-flex align-items-center justify-content-center" 
                                         style="width: 80px; height: 80px;">
                                        <i class="fas fa-store fa-2x text-muted"></i>
                                    </div>
                                <?php endif; ?>
                                
                                <?php if ($stand['event_nom']): ?>
                                    <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-info">
                                        <i class="fas fa-calendar-alt"></i>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <div>
                                <h6 class="mb-1 fw-semibold"><?= e($stand['nom']) ?></h6>
                                
                                <?php if ($stand['client_societe']): ?>
                                    <div class="d-flex align-items-center mb-1">
                                        <i class="fas fa-building text-success me-2"></i>
                                        <span class="text-success fw-medium"><?= e($stand['client_societe']) ?></span>
                                        <?php if ($stand['client_pays']): ?>
                                            <span class="badge bg-success bg-opacity-25 text-success ms-2">
                                                <?= e($stand['client_pays']) ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                    <?php if ($stand['contact_commercial_prenom'] && $stand['contact_commercial_nom']): ?>
                                        <small class="text-muted">
                                            <i class="fas fa-user me-1"></i>
                                            <?= e($stand['contact_commercial_prenom'] . ' ' . $stand['contact_commercial_nom']) ?>
                                            <?php if ($stand['contact_commercial_mail']): ?>
                                                <a href="mailto:<?= e($stand['contact_commercial_mail']) ?>" 
                                                   class="ms-1" title="Envoyer un email">
                                                    <i class="fas fa-envelope text-primary"></i>
                                                </a>
                                            <?php endif; ?>
                                        </small>
                                    <?php endif; ?>
                                <?php else: ?>
                                    <small class="text-warning">
                                        <i class="fas fa-exclamation-triangle me-1"></i>Aucun client assigné
                                    </small>
                                <?php endif; ?>

                                <?php if ($stand['event_nom']): ?>
                                    <div class="mt-1">
                                        <small class="text-primary">
                                            <i class="fas fa-calendar-alt me-1"></i>
                                            <strong><?= e($stand['event_nom']) ?></strong>
                                            <?php if ($stand['event_lieu']): ?>
                                                - <?= e($stand['event_lieu']) ?>
                                            <?php endif; ?>
                                            <br>
                                            <span class="text-muted">
                                                <?= formatDate($stand['event_date_debut']) ?> 
                                                → <?= formatDate($stand['event_date_fin']) ?>
                                            </span>
                                        </small>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <div class="text-center">
                                <span class="badge <?= !empty($stand['is_custom']) ? 'bg-info' : 'bg-success' ?> fs-6">
                                    <?= getCategorieFromSurface($stand['surface']) ?>
                                </span>
                                <?php if (!empty($stand['is_custom'])): ?>
                                    <div><small class="text-muted">Personnalisé</small></div>
                                <?php else: ?>
                                    <div><small class="text-muted">Standard</small></div>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td class="text-center">
                            <div class="d-flex flex-column align-items-center">
                                <span class="fw-bold text-primary"><?= !empty($stand['surface']) ? $stand['surface'] . ' m²' : '-' ?></span>
                                <?php if ($stand['nb_angles'] > 0): ?>
                                    <small class="text-muted">
                                        <i class="fas fa-vector-square me-1"></i><?= $stand['nb_angles'] ?> angle<?= $stand['nb_angles'] > 1 ? 's' : '' ?>
                                    </small>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td class="text-center">
                            <span class="badge bg-secondary fs-6">
                                <i class="fas fa-puzzle-piece me-1"></i>
                                <?= $stand['nb_pieces'] ?> 
                            </span>
                            <div><small class="text-muted">pièce<?= $stand['nb_pieces'] > 1 ? 's' : '' ?></small></div>
                        </td>
                        <td class="text-end">
                            <div class="d-flex flex-column align-items-end">
                                <span class="fw-bold text-dark fs-5">
                                    <?= !empty($stand['prix_total_ht']) ? number_format($stand['prix_total_ht'], 2, ',', ' ') . ' €' : '-' ?>
                                </span>
                                <small class="text-muted">HT</small>
                                <?php if (!empty($stand['prix_total_ht']) && $stand['prix_total_ht'] > $stand['prix_location_ht']): ?>
                                    <small class="text-success">
                                        <i class="fas fa-plus-circle me-1"></i>+ Options
                                    </small>
                                <?php endif; ?>
                            </div>
                        </td>
                        <td>
                            <div class="btn-group">
                                <a href="gestion_stand.php?action=details&id=<?= $stand['id'] ?>" 
                                   class="btn btn-outline-primary btn-sm"
                                   title="Voir les détails">
                                    <i class="fas fa-eye"></i>
                                </a>
                                <a href="gestion_stand.php?action=modifier&id=<?= $stand['id'] ?>" 
                                   class="btn btn-outline-success btn-sm"
                                   title="Modifier">
                                    <i class="fas fa-edit"></i>
                                </a>
                                <div class="dropdown">
                                    <button class="btn btn-outline-secondary btn-sm dropdown-toggle" 
                                            type="button" data-bs-toggle="dropdown">
                                        <i class="fas fa-ellipsis-v"></i>
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="duplicate.php?id=<?= $stand['id'] ?>">
                                            <i class="fas fa-copy text-info me-2"></i>Dupliquer
                                        </a></li>
                                        <li><a class="dropdown-item" href="export_stand.php?id=<?= $stand['id'] ?>">
                                            <i class="fas fa-file-pdf text-danger me-2"></i>Export PDF
                                        </a></li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li><button class="dropdown-item text-danger" 
                                                    data-bs-toggle="modal"
                                                    data-bs-target="#deleteModal"
                                                    data-id="<?= $stand['id'] ?>"
                                                    data-name="<?= e($stand['nom']) ?>">
                                            <i class="fas fa-trash me-2"></i>Supprimer
                                        </button></li>
                                    </ul>
                                </div>
                            </div>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

    <!-- Vue Grille -->
    <div id="gridView" class="row g-4" style="display: none;">
        <?php foreach ($stands as $stand): ?>
            <div class="col-xl-4 col-lg-6 stand-card" 
                 data-event-id="<?= $stand['event_id'] ?? '' ?>"
                 data-event-name="<?= e($stand['event_nom'] ?? '') ?>"
                 data-client-id="<?= $stand['client_id'] ?? '' ?>"
                 data-client-name="<?= e($stand['client_societe'] ?? '') ?>"
                 data-surface="<?= $stand['surface'] ?? 0 ?>"
                 data-prix="<?= $stand['prix_total_ht'] ?? 0 ?>"
                 data-type="<?= !empty($stand['is_custom']) ? 'personnalise' : 'preconfigure' ?>">
                <div class="card h-100 shadow-sm stand-card-hover">
                    
                    <!-- Header avec badges -->
                    <div class="card-header bg-white border-bottom-0 pb-0">
                        <div class="d-flex justify-content-between align-items-start mb-2">
                            <span class="badge <?= !empty($stand['is_custom']) ? 'bg-info' : 'bg-success' ?>">
                                <?= getCategorieFromSurface($stand['surface']) ?>
                            </span>
                            <?php if ($stand['event_nom']): ?>
                                <span class="badge bg-primary">
                                    <i class="fas fa-calendar-alt me-1"></i>Événement
                                </span>
                            <?php endif; ?>
                        </div>
                        <h5 class="card-title mb-1"><?= e($stand['nom']) ?></h5>
                        
                        <?php if ($stand['client_societe']): ?>
                            <div class="text-success mb-2">
                                <i class="fas fa-building me-1"></i>
                                <strong><?= e($stand['client_societe']) ?></strong>
                                <?php if ($stand['client_pays']): ?>
                                    <small class="text-muted">(<?= e($stand['client_pays']) ?>)</small>
                                <?php endif; ?>
                            </div>
                        <?php else: ?>
                            <div class="text-warning mb-2">
                                <i class="fas fa-exclamation-triangle me-1"></i>
                                <small>Aucun client assigné</small>
                            </div>
                        <?php endif; ?>
                    </div>

                    <!-- Image -->
                    <?php if (!empty($stand['image'])): ?>
                        <img src="<?= url('assets/uploads/stands/' . $stand['image']) ?>" 
                             class="card-img-top" 
                             alt="<?= e($stand['nom']) ?>"
                             style="height: 200px; object-fit: cover;">
                    <?php else: ?>
                        <div class="card-img-top bg-light text-center d-flex align-items-center justify-content-center" 
                             style="height: 200px;">
                            <i class="fas fa-store fa-4x text-muted"></i>
                        </div>
                    <?php endif; ?>
                    
                    <!-- Contenu -->
                    <div class="card-body">
                        <!-- Spécifications techniques -->
                        <div class="row text-center mb-3">
                            <div class="col-4">
                                <div class="text-primary">
                                    <i class="fas fa-expand-arrows-alt d-block mb-1"></i>
                                    <strong><?= $stand['surface'] ?></strong><br>
                                    <small class="text-muted">m²</small>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="text-secondary">
                                    <i class="fas fa-puzzle-piece d-block mb-1"></i>
                                    <strong><?= $stand['nb_pieces'] ?></strong><br>
                                    <small class="text-muted">pièces</small>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="text-success">
                                    <i class="fas fa-euro-sign d-block mb-1"></i>
                                    <strong><?= number_format($stand['prix_total_ht'], 0, ',', ' ') ?></strong><br>
                                    <small class="text-muted">€ HT</small>
                                </div>
                            </div>
                        </div>

                        <!-- Informations événement -->
                        <?php if ($stand['event_nom']): ?>
                            <div class="alert alert-primary py-2 mb-3">
                                <small>
                                    <strong><i class="fas fa-calendar-alt me-1"></i><?= e($stand['event_nom']) ?></strong><br>
                                    <?= formatDate($stand['event_date_debut']) ?> → <?= formatDate($stand['event_date_fin']) ?>
                                    <?php if ($stand['event_lieu']): ?>
                                        <br><i class="fas fa-map-marker-alt me-1"></i><?= e($stand['event_lieu']) ?>
                                    <?php endif; ?>
                                </small>
                            </div>
                        <?php endif; ?>

                        <!-- Description -->
                        <?php if (!empty($stand['description'])): ?>
                            <p class="card-text text-muted small">
                                <?= substr(e($stand['description']), 0, 120) ?>
                                <?= strlen($stand['description']) > 120 ? '...' : '' ?>
                            </p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Footer avec actions -->
                    <div class="card-footer bg-white border-top">
                        <div class="d-grid gap-2">
                            <div class="btn-group">
                                <a href="gestion_stand.php?action=details&id=<?= $stand['id'] ?>" 
                                   class="btn btn-outline-primary btn-sm">
                                    <i class="fas fa-eye me-1"></i>Détails
                                </a>
                                <a href="gestion_stand.php?action=modifier&id=<?= $stand['id'] ?>" 
                                   class="btn btn-outline-success btn-sm">
                                    <i class="fas fa-edit me-1"></i>Modifier
                                </a>
                                <div class="dropdown">
                                    <button class="btn btn-outline-secondary btn-sm dropdown-toggle" 
                                            type="button" data-bs-toggle="dropdown">
                                        <i class="fas fa-ellipsis-v"></i>
                                    </button>
                                    <ul class="dropdown-menu">
                                        <li><a class="dropdown-item" href="duplicate.php?id=<?= $stand['id'] ?>">
                                            <i class="fas fa-copy me-2"></i>Dupliquer
                                        </a></li>
                                        <li><a class="dropdown-item" href="export_stand.php?id=<?= $stand['id'] ?>">
                                            <i class="fas fa-download me-2"></i>Export
                                        </a></li>
                                        <li><hr class="dropdown-divider"></li>
                                        <li><button class="dropdown-item text-danger" 
                                                    data-bs-toggle="modal"
                                                    data-bs-target="#deleteModal"
                                                    data-id="<?= $stand['id'] ?>"
                                                    data-name="<?= e($stand['nom']) ?>">
                                            <i class="fas fa-trash me-2"></i>Supprimer
                                        </button></li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Modal de suppression -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title">
                    <i class="fas fa-exclamation-triangle me-2"></i>Confirmer la suppression
                </h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Attention :</strong> Cette action est irréversible.
                </div>
                <p>Voulez-vous vraiment supprimer le stand <strong id="standName"></strong> ?</p>
                <p class="text-muted small">
                    Les liens avec les événements seront également supprimés, mais les pièces du catalogue resteront intactes.
                </p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Annuler
                </button>
                <form action="supprimer.php" method="POST" class="d-inline">
                    <input type="hidden" name="id" id="standId">
                    <button type="submit" class="btn btn-danger">
                        <i class="fas fa-trash me-2"></i>Supprimer définitivement
                    </button>
                </form>
            </div>
        </div>
    </div>
</div>

<style>
/* Animations et transitions */
.fade-in {
    animation: fadeIn 0.6s ease-out;
}

@keyframes fadeIn {
    from { 
        opacity: 0; 
        transform: translateY(-20px); 
    }
    to { 
        opacity: 1; 
        transform: translateY(0); 
    }
}

/* Cartes statistiques */
.icon-wrapper {
    min-width: 50px;
    min-height: 50px;
    display: flex;
    align-items: center;
    justify-content: center;
}

/* En-têtes dégradés */
.bg-gradient-primary {
    background: linear-gradient(135deg, #0d6efd, #6f42c1) !important;
}

/* Cartes stands hover */
.stand-card-hover {
    transition: all 0.3s ease;
}

.stand-card-hover:hover {
    transform: translateY(-8px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15) !important;
}

/* Tableau amélioré */
.table-dark th {
    border-color: #495057;
    font-weight: 600;
}

.stand-row {
    transition: all 0.2s ease;
}

.stand-row:hover {
    background-color: #f8f9fa;
    transform: scale(1.01);
    box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
}

/* Boutons sortables */
.sortable {
    cursor: pointer;
    user-select: none;
}

.sortable:hover {
    background-color: rgba(255, 255, 255, 0.1);
}

.sortable.active {
    background-color: rgba(255, 255, 255, 0.2);
}

/* Badges améliorés */
.badge {
    font-weight: 500;
}

/* Filtres avancés */
.form-control:focus, .form-select:focus {
    border-color: #0d6efd;
    box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
}

/* Vue grille */
.stand-card {
    transition: all 0.3s ease;
}

.stand-card:hover {
    transform: translateY(-5px);
}

/* Responsive */
@media (max-width: 768px) {
    .d-flex.justify-content-between {
        flex-direction: column;
        gap: 1rem;
    }
    
    .btn-group {
        width: 100%;
    }
    
    .table-responsive {
        font-size: 0.875rem;
    }
}

/* Loading states */
.loading {
    opacity: 0.6;
    pointer-events: none;
}

/* Selection multiple */
.stand-row.selected {
    background-color: #e3f2fd !important;
}

.stand-card.selected .card {
    border-color: #2196f3 !important;
    box-shadow: 0 0 0 2px rgba(33, 150, 243, 0.2) !important;
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Éléments DOM
    const searchInput = document.getElementById('searchInput');
    const clientFilter = document.getElementById('clientFilter');
    const eventFilter = document.getElementById('eventFilter');
    const typeFilter = document.getElementById('typeFilter');
    const statutFilter = document.getElementById('statutFilter');
    const surfaceMin = document.getElementById('surfaceMin');
    const surfaceMax = document.getElementById('surfaceMax');
    const prixMin = document.getElementById('prixMin');
    const prixMax = document.getElementById('prixMax');
    const clearSearchBtn = document.getElementById('clearSearch');
    const resetButton = document.getElementById('resetFilters');
    const viewToggle = document.getElementById('viewToggle');
    const gridView = document.getElementById('gridView');
    const tableView = document.getElementById('tableView');
    const table = document.getElementById('standsTable');
    const headers = table.querySelectorAll('th.sortable');
    const tbody = table.querySelector('tbody');
    const resultCounter = document.getElementById('resultCounter');
    const selectAllCheckbox = document.getElementById('selectAll');
    const eventStats = document.getElementById('eventStats');
    const eventActions = document.getElementById('eventActions');
    
    // Variables globales
    let isGridView = false;
    let currentSort = { column: '', direction: 'asc' };
    let selectedStands = new Set();

    // Message "Aucun résultat"
    const noResultsRow = document.createElement('tr');
    noResultsRow.innerHTML = `
        <td colspan="8" class="text-center py-5">
            <div class="text-muted">
                <i class="fas fa-search fa-3x mb-3"></i>
                <h5>Aucun stand trouvé</h5>
                <p>Modifiez vos critères de recherche pour voir plus de résultats</p>
                <button class="btn btn-outline-primary" onclick="resetAllFilters()">
                    <i class="fas fa-undo me-2"></i>Réinitialiser les filtres
                </button>
            </div>
        </td>
    `;

    // Fonction de mise à jour du compteur enrichie
    function updateCounter(items) {
        const categoryCounts = {};
        const clientCounts = {};
        const eventCounts = {};
        let totalCA = 0;
        let totalPieces = 0;
        let standsWithClient = 0;
        let standsWithEvent = 0;
        
        items.forEach(item => {
            // Compteurs par catégorie
            const category = isGridView ? 
                item.querySelector('.badge').textContent.trim() :
                item.querySelector('td:nth-child(4) .badge').textContent.trim();
            
            const clientName = item.getAttribute('data-client-name');
            const eventName = item.getAttribute('data-event-name');
            const prix = parseFloat(item.getAttribute('data-prix')) || 0;
            
            if (!categoryCounts[category]) categoryCounts[category] = 0;
            categoryCounts[category]++;
            
            if (clientName) {
                standsWithClient++;
                if (!clientCounts[clientName]) clientCounts[clientName] = 0;
                clientCounts[clientName]++;
            }
            
            if (eventName) {
                standsWithEvent++;
                if (!eventCounts[eventName]) eventCounts[eventName] = 0;
                eventCounts[eventName]++;
            }
            
            totalCA += prix;
            
            // Comptage des pièces
            const piecesText = isGridView ? 
                item.querySelector('.text-secondary strong')?.textContent :
                item.querySelector('.badge.bg-secondary')?.textContent;
            if (piecesText) {
                totalPieces += parseInt(piecesText) || 0;
            }
        });

        // Message principal
        let counterHTML = `<strong>${items.length}</strong> stand${items.length > 1 ? 's' : ''} affiché${items.length > 1 ? 's' : ''}`;
        
        if (items.length > 0) {
            // Répartition par catégorie
            counterHTML += ' <span class="text-muted">(';
            const categoryDetails = Object.entries(categoryCounts)
                .sort((a, b) => b[1] - a[1])
                .map(([category, count]) => `${count} en ${category}`)
                .join(', ');
            counterHTML += categoryDetails + ')</span>';

            // Informations complémentaires
            const details = [];
            if (standsWithClient > 0) {
                details.push(`${standsWithClient} avec client`);
            }
            if (standsWithEvent > 0) {
                details.push(`${standsWithEvent} avec événement`);
            }
            if (totalCA > 0) {
                details.push(`${Math.round(totalCA).toLocaleString('fr-FR')}€ HT total`);
            }
            if (totalPieces > 0) {
                details.push(`${totalPieces} pièces total`);
            }
            
            if (details.length > 0) {
                counterHTML += `<br><small class="text-muted">${details.join(' • ')}</small>`;
            }
        }

        if (resultCounter) {
            resultCounter.innerHTML = counterHTML;
        }
        
        // Mise à jour des stats d'événement si applicable
        if (eventFilter.value) {
            updateEventStatsFromFilteredItems(items);
        }
    }

    // Fonction de mise à jour des stats d'événement
    function updateEventStatsFromFilteredItems(items) {
        const eventId = eventFilter.value;
        const eventStands = items.filter(item => 
            item.getAttribute('data-event-id') === eventId && 
            item.style.display !== 'none'
        );
        
        let totalSurface = 0;
        let totalPieces = 0;
        let totalCA = 0;
        let typesCount = { preconfigure: 0, personnalise: 0 };
        
        eventStands.forEach(stand => {
            totalSurface += parseFloat(stand.getAttribute('data-surface')) || 0;
            totalCA += parseFloat(stand.getAttribute('data-prix')) || 0;
            
            const type = stand.getAttribute('data-type');
            if (type) typesCount[type]++;
            
            // Comptage des pièces
            const piecesText = isGridView ? 
                stand.querySelector('.text-secondary strong')?.textContent :
                stand.querySelector('.badge.bg-secondary')?.textContent;
            if (piecesText) {
                totalPieces += parseInt(piecesText) || 0;
            }
        });
        
        // Mise à jour de l'affichage
        document.getElementById('eventStandsCount').textContent = eventStands.length;
        document.getElementById('eventSurfaceTotal').textContent = Math.round(totalSurface);
        document.getElementById('eventCA').textContent = Math.round(totalCA).toLocaleString('fr-FR') + '€';
        document.getElementById('eventCAMoyenne').textContent = eventStands.length > 0 ? Math.round(totalCA / eventStands.length) : 0;
        document.getElementById('eventPiecesTotal').textContent = totalPieces;
        
        // Types de stands
        const typesText = [];
        if (typesCount.preconfigure > 0) typesText.push(`${typesCount.preconfigure} standards`);
        if (typesCount.personnalise > 0) typesText.push(`${typesCount.personnalise} personnalisés`);
        document.getElementById('eventStandsTypes').textContent = typesText.join(', ') || 'Aucun';
        
        // Status des pièces
        document.getElementById('eventPiecesStatus').textContent = `${totalPieces} pièces à préparer`;
        
        // Barres de progression
        document.getElementById('eventStandsProgress').style.width = '100%';
        document.getElementById('eventPiecesProgress').style.width = '85%';
    }

    // Fonction de filtrage avancée
    function filterStands() {
        const searchTerm = searchInput.value.toLowerCase();
        const clientValue = clientFilter.value;
        const eventValue = eventFilter.value;
        const typeValue = typeFilter.value;
        const statutValue = statutFilter.value;
        const minSurface = parseFloat(surfaceMin.value) || 0;
        const maxSurface = parseFloat(surfaceMax.value) || Infinity;
        const minPrix = parseFloat(prixMin.value) || 0;
        const maxPrix = parseFloat(prixMax.value) || Infinity;
        
        let visibleItems = [];

        const items = isGridView ? 
            Array.from(gridView.querySelectorAll('.stand-card')) :
            Array.from(tbody.querySelectorAll('tr.stand-row'));

        items.forEach(item => {
            const surface = parseFloat(item.getAttribute('data-surface')) || 0;
            const prix = parseFloat(item.getAttribute('data-prix')) || 0;
            const type = item.getAttribute('data-type');
            const clientId = item.getAttribute('data-client-id');
            const eventId = item.getAttribute('data-event-id');
            
            // Texte pour recherche globale
            const searchableText = item.textContent.toLowerCase();

            // Filtres
            const matchSearch = !searchTerm || searchableText.includes(searchTerm);
            const matchClient = !clientValue || 
                (clientValue === '__non_assigne__' ? !clientId : clientId === clientValue);
            const matchEvent = !eventValue || eventId === eventValue;
            const matchType = !typeValue || type === typeValue;
            const matchStatut = !statutValue || 
                (statutValue === 'avec_client' ? clientId : 
                 statutValue === 'sans_client' ? !clientId :
                 statutValue === 'avec_evenement' ? eventId : true);
            const matchSurface = surface >= minSurface && surface <= maxSurface;
            const matchPrix = prix >= minPrix && prix <= maxPrix;
            
            const isVisible = matchSearch && matchClient && matchEvent && 
                            matchType && matchStatut && matchSurface && matchPrix;
            
            item.style.display = isVisible ? '' : 'none';
            if (isVisible) visibleItems.push(item);
        });

        // Mise à jour du compteur et du message vide
        updateCounter(visibleItems);

        if (visibleItems.length === 0 && !isGridView) {
            if (!tbody.contains(noResultsRow)) {
                tbody.appendChild(noResultsRow);
            }
        } else if (!isGridView && tbody.contains(noResultsRow)) {
            tbody.removeChild(noResultsRow);
        }
    }

    // Gestion événement
    eventFilter.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        
        if (this.value) {
            // Afficher les statistiques de l'événement
            showEventStats(selectedOption);
            eventStats.style.display = 'flex';
            eventActions.style.display = 'flex';
            
            // Configuration des boutons d'action
            const generateBtn = document.getElementById('generateInventoryBtn');
            const exportBtn = document.getElementById('exportEventBtn');
            
            if (generateBtn) {
                generateBtn.onclick = function() {
                    window.location.href = `generate_all_stands_inventory.php?event_id=${eventFilter.value}`;
                };
            }
            
            if (exportBtn) {
                exportBtn.onclick = function() {
                    window.location.href = `export.php?format=excel&event_id=${eventFilter.value}`;
                };
            }
        } else {
            // Cacher les statistiques si aucun événement sélectionné
            eventStats.style.display = 'none';
            eventActions.style.display = 'none';
        }
        
        filterStands();
    });

    // Nouvelle fonction pour afficher les stats d'événement
    function showEventStats(selectedOption) {
        const eventName = selectedOption.textContent.split(' -')[0].trim();
        const lieu = selectedOption.dataset.lieu;
        const debut = selectedOption.dataset.debut;
        const fin = selectedOption.dataset.fin;
        
        // Mise à jour des informations générales
        document.getElementById('eventName').textContent = eventName;
        document.getElementById('eventLieu').textContent = lieu;
        document.getElementById('eventDates').textContent = `${formatDate(debut)} → ${formatDate(fin)}`;
    }

    // Event Listeners pour les filtres
    searchInput.addEventListener('input', filterStands);
    clientFilter.addEventListener('change', filterStands);
    typeFilter.addEventListener('change', filterStands);
    statutFilter.addEventListener('change', filterStands);
    surfaceMin.addEventListener('input', filterStands);
    surfaceMax.addEventListener('input', filterStands);
    prixMin.addEventListener('input', filterStands);
    prixMax.addEventListener('input', filterStands);

    clearSearchBtn.addEventListener('click', function() {
        searchInput.value = '';
        filterStands();
    });

    resetButton.addEventListener('click', function() {
        searchInput.value = '';
        clientFilter.value = '';
        eventFilter.value = '';
        typeFilter.value = '';
        statutFilter.value = '';
        surfaceMin.value = '';
        surfaceMax.value = '';
        prixMin.value = '';
        prixMax.value = '';
        currentSort = { column: '', direction: 'asc' };
        headers.forEach(h => {
            h.querySelector('i').className = 'fas fa-sort me-1';
            h.classList.remove('active');
        });
        selectedStands.clear();
        document.querySelectorAll('.stand-checkbox').forEach(cb => cb.checked = false);
        selectAllCheckbox.checked = false;
        
        // Cacher les statistiques d'événement
        eventStats.style.display = 'none';
        eventActions.style.display = 'none';
        
        filterStands();
    });

    // Gestion vue grille/liste
    viewToggle.addEventListener('click', function() {
        isGridView = !isGridView;
        this.innerHTML = isGridView ? 
            '<i class="fas fa-list me-2"></i>Vue liste' : 
            '<i class="fas fa-th me-2"></i>Vue grille';
        
        gridView.style.display = isGridView ? 'flex' : 'none';
        tableView.style.display = isGridView ? 'none' : 'block';
        
        filterStands();
    });

    // Gestion de la modal de suppression
    const deleteModal = document.getElementById('deleteModal');
    if (deleteModal) {
        deleteModal.addEventListener('show.bs.modal', function(event) {
            const button = event.relatedTarget;
            const id = button.getAttribute('data-id');
            const name = button.getAttribute('data-name');
            
            deleteModal.querySelector('#standId').value = id;
            deleteModal.querySelector('#standName').textContent = name;
        });
    }

    // Fonction globale pour afficher tous les stands
    window.showAllStands = function() {
        eventFilter.value = '';
        eventStats.style.display = 'none';
        eventActions.style.display = 'none';
        filterStands();
    };

    // Fonction globale pour reset
    window.resetAllFilters = function() {
        resetButton.click();
    };

    // Fonction helper pour formater les dates (côté client)
    function formatDate(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        return date.toLocaleDateString('fr-FR', { day: '2-digit', month: '2-digit', year: 'numeric' });
    }

    // Initialisation
    filterStands();
});
</script>

<?php require_once '../includes/footer.php'; ?>