﻿<?php
require_once '../includes/init.php';
requireLogin();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    redirect('/stand/creation.php');
}

try {
    $db->beginTransaction();

    // Validation des données du formulaire
    $errors = [];
    $is_custom = !empty($_POST['is_custom']) && $_POST['is_custom'] == '1';
    
    if (empty($_POST['nom'])) {
        $errors[] = "Le nom du stand est requis";
    }
    
    if (empty($_POST['client_id']) || !is_numeric($_POST['client_id'])) {
        $errors[] = "Le client est requis";
    }
    
    // Validation du type de stand uniquement si non personnalisé
    if (!$is_custom && empty($_POST['type_id'])) {
        $errors[] = "Le type de stand est requis pour un stand pré-configuré";
    }
    
    if (empty($_POST['surface']) || !is_numeric($_POST['surface']) || $_POST['surface'] <= 0) {
        $errors[] = "La surface doit être un nombre positif";
    }
    
    if (empty($_POST['prix_location_ht']) || !is_numeric($_POST['prix_location_ht']) || $_POST['prix_location_ht'] < 0) {
        $errors[] = "Le prix de location doit être un nombre positif ou nul";
    }

    // Validation du code porte (optionnel mais doit être 3 chiffres si renseigné)
    if (!empty($_POST['code_porte']) && !preg_match('/^[0-9]{3}$/', $_POST['code_porte'])) {
        $errors[] = "Le code porte doit contenir exactement 3 chiffres";
    }

    // Validation du côté de la poignée (optionnel mais doit être valide si renseigné)
    if (!empty($_POST['cote_poignee_porte']) && !in_array($_POST['cote_poignee_porte'], ['droite', 'gauche'])) {
        $errors[] = "Le côté de la poignée doit être 'droite' ou 'gauche'";
    }

    // Validation du nombre de plaques PVC (doit être positif ou nul)
    if (isset($_POST['nb_plaques_pvc_supplementaires']) && (!is_numeric($_POST['nb_plaques_pvc_supplementaires']) || $_POST['nb_plaques_pvc_supplementaires'] < 0)) {
        $errors[] = "Le nombre de plaques PVC doit être un nombre positif ou nul";
    }

    // Validation du client
    if (!empty($_POST['client_id'])) {
        $stmt = $db->prepare("SELECT id, societe FROM clients WHERE id = ?");
        $stmt->execute([$_POST['client_id']]);
        $client = $stmt->fetch();
        if (!$client) {
            $errors[] = "Le client sélectionné n'existe pas";
        }
    }

    // Validation de l'événement si sélectionné
    if (!empty($_POST['evenement_id'])) {
        $stmt = $db->prepare("
            SELECT id, date_fin 
            FROM events 
            WHERE id = ? 
            AND date_fin >= CURRENT_DATE
            AND statut != 'annule'
        ");
        $stmt->execute([$_POST['evenement_id']]);
        if (!$stmt->fetch()) {
            $errors[] = "L'événement sélectionné n'est pas valide";
        }
    }

    if (!empty($errors)) {
        throw new Exception(implode("\n", $errors));
    }

    // Gestion de l'image
    $image_filename = null;
    if (!empty($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        // Créer le dossier s'il n'existe pas
        $upload_dir = '../assets/uploads/stands/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        $file_extension = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));
        $new_filename = 'stand_' . uniqid() . '.' . $file_extension;
        
        if (!in_array($file_extension, ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
            throw new Exception("Format d'image non supporté. Formats acceptés : JPG, PNG, GIF, WebP");
        }
        
        if ($_FILES['image']['size'] > 5 * 1024 * 1024) { // 5MB
            throw new Exception("L'image est trop volumineuse (max 5MB)");
        }
        
        if (!move_uploaded_file($_FILES['image']['tmp_name'], $upload_dir . $new_filename)) {
            throw new Exception("Erreur lors de l'upload de l'image");
        }
        
        $image_filename = $new_filename;
    }

    // Insertion du stand avec TOUS les nouveaux champs
    $stmt = $db->prepare("
        INSERT INTO stands (
            nom, 
            client_id,
            type_id, 
            surface, 
            nb_angles, 
            prix_location_ht, 
            description, 
            image,
            is_custom,
            code_porte,
            cote_poignee_porte,
            nb_plaques_pvc_supplementaires,
            remarques,
            created_at, 
            updated_at
        ) VALUES (
            :nom,
            :client_id,
            :type_id,
            :surface,
            :nb_angles,
            :prix_location_ht,
            :description,
            :image,
            :is_custom,
            :code_porte,
            :cote_poignee_porte,
            :nb_plaques_pvc_supplementaires,
            :remarques,
            NOW(),
            NOW()
        )
    ");

    $stmt->execute([
        'nom' => $_POST['nom'],
        'client_id' => $_POST['client_id'],
        'type_id' => $is_custom ? null : $_POST['type_id'],
        'surface' => $_POST['surface'],
        'nb_angles' => $_POST['nb_angles'] ?? 0,
        'prix_location_ht' => $_POST['prix_location_ht'],
        'description' => $_POST['description'] ?? null,
        'image' => $image_filename,
        'is_custom' => $is_custom ? 1 : 0,
        'code_porte' => !empty($_POST['code_porte']) ? $_POST['code_porte'] : null,
        'cote_poignee_porte' => !empty($_POST['cote_poignee_porte']) ? $_POST['cote_poignee_porte'] : null,
        'nb_plaques_pvc_supplementaires' => isset($_POST['nb_plaques_pvc_supplementaires']) ? (int)$_POST['nb_plaques_pvc_supplementaires'] : 0,
        'remarques' => !empty($_POST['remarques']) ? $_POST['remarques'] : null
    ]);

    $stand_id = $db->lastInsertId();

    // Si un événement est sélectionné, créer l'association
    if (!empty($_POST['evenement_id'])) {
        $stmt = $db->prepare("
            INSERT INTO event_stands (
                event_id, 
                stand_id, 
                created_at
            ) VALUES (?, ?, NOW())
        ");
        $stmt->execute([$_POST['evenement_id'], $stand_id]);
        $event_stand_id = $db->lastInsertId();

        // Si des options ont été sélectionnées, les associer à l'event_stand
        if (!empty($_POST['options']) && is_array($_POST['options'])) {
            $stmt = $db->prepare("
                INSERT INTO event_stand_options (
                    event_stand_id,
                    option_id,
                    quantite
                ) VALUES (?, ?, 1)
            ");
            
            foreach ($_POST['options'] as $option_id) {
                // Vérifier que l'option existe
                $checkStmt = $db->prepare("SELECT id FROM stand_options WHERE id = ? AND is_active = 1");
                $checkStmt->execute([$option_id]);
                if ($checkStmt->fetch()) {
                    $stmt->execute([$event_stand_id, $option_id]);
                }
            }
        }
    }

    // Log de l'action avec détails des nouveaux champs
    $stmt = $db->prepare("
        INSERT INTO activity_logs (
            user_id, 
            action, 
            details, 
            created_at
        ) VALUES (?, 'CREATE', ?, NOW())
    ");

    $standType = $is_custom ? "personnalisé" : "pré-configuré";
    $clientInfo = isset($client) ? " pour " . $client['societe'] : "";
    $technicalInfo = "";
    
    // Ajouter les détails techniques dans le log
    if (!empty($_POST['code_porte'])) {
        $technicalInfo .= " - Code porte: " . $_POST['code_porte'];
    }
    if (!empty($_POST['cote_poignee_porte'])) {
        $technicalInfo .= " - Poignée: " . $_POST['cote_poignee_porte'];
    }
    if (!empty($_POST['nb_plaques_pvc_supplementaires']) && $_POST['nb_plaques_pvc_supplementaires'] > 0) {
        $technicalInfo .= " - " . $_POST['nb_plaques_pvc_supplementaires'] . " plaque(s) PVC sup.";
    }
    
    $logDetails = "Création du stand '{$_POST['nom']}' ({$standType}){$clientInfo}{$technicalInfo}";
    $stmt->execute([$_SESSION['user_id'], $logDetails]);

    $db->commit();
    
    // Message de succès avec détails
    $successMessage = "Le stand a été créé avec succès et associé au client.";
    if (!empty($_POST['code_porte'])) {
        $successMessage .= " Code porte: " . $_POST['code_porte'] . ".";
    }
    
    addMessage($successMessage, "success");
    redirect('/stand/liste.php');

} catch (Exception $e) {
    // Si une erreur survient, supprimer l'image si elle a été uploadée
    if (isset($image_filename) && $image_filename && file_exists('../assets/uploads/stands/' . $image_filename)) {
        unlink('../assets/uploads/stands/' . $image_filename);
    }
    
    $db->rollBack();
    error_log("Erreur lors de la création du stand : " . $e->getMessage());
    addMessage("Erreur lors de la création du stand : " . $e->getMessage(), "error");
    
    // Conserver les données du formulaire en session pour éviter de les perdre
    $_SESSION['form_data'] = $_POST;
    redirect('/stand/creation.php');
}
?>